import { NextResponse } from 'next/server';
import axios, { AxiosError, AxiosResponse } from 'axios';
import { Supplier } from '@/app/types/Supplier';

// Use NEXT_PUBLIC_API_BASE_URL from .env.local
const API_BASE_URL = process.env.NEXT_PUBLIC_API_BASE_URL || 'https://api.qms-dev.iatsolutions.co';

// Interface for API responses
interface SupplierResponse {
  Message: string;
  Detail?: Supplier;
  Suppliers?: Supplier[];
}

// Utility to convert object to URLSearchParams
const toFormData = (data: Record<string, unknown>) => {
  const formData = new URLSearchParams();
  Object.entries(data).forEach(([key, value]) => {
    if (value !== undefined && value !== null) {
      formData.append(key, value.toString());
    }
  });
  return formData;
};

// Utility to get Authorization header from request
const getAuthHeaders = (request: Request) => {
  const authHeader = request.headers.get('Authorization') || '';
  return {
    Authorization: authHeader,
    Accept: 'application/json',
  };
};

// Utility to format error response
const showAxiosError = (error: AxiosError<unknown> | null | undefined, fallback: string) => {
  let detail: string;
  if (error instanceof AxiosError) {
    const responseData = error.response?.data as { detail?: string; error?: string } | undefined;
    detail = responseData?.detail || responseData?.error || error.message || fallback;
  } else {
    detail = fallback;
  }
  return { error: detail, status: error?.response?.status || 500 };
};

// GET all suppliers
export async function GET(request: Request) {
  try {
    const headers = getAuthHeaders(request);
    if (!headers.Authorization) {
      const errorResponse = showAxiosError(null, 'Authentication token missing');
      return NextResponse.json({ error: errorResponse.error }, { status: 401 });
    }

    const response: AxiosResponse<SupplierResponse> = await axios.get(`${API_BASE_URL}/view-all-suppliers`, {
      headers,
    });

    return NextResponse.json(response.data);
  } catch (error) {
    const axiosError = error as AxiosError<unknown>;
    const errorResponse = showAxiosError(axiosError, 'Failed to fetch suppliers');
    return NextResponse.json({ error: errorResponse.error }, { status: errorResponse.status });
  }
}

// POST new supplier
export async function POST(request: Request) {
  try {
    const headers = getAuthHeaders(request);
    if (!headers.Authorization) {
      const errorResponse = showAxiosError(null, 'Authentication token missing');
      return NextResponse.json({ error: errorResponse.error }, { status: 401 });
    }

    const body = await request.json();
    const response: AxiosResponse<SupplierResponse> = await axios.post(
      `${API_BASE_URL}/create-supplier`,
      toFormData(body),
      {
        headers: {
          ...headers,
          'Content-Type': 'application/x-www-form-urlencoded',
        },
      }
    );

    return NextResponse.json(response.data);
  } catch (error) {
    const axiosError = error as AxiosError<unknown>;
    const errorResponse = showAxiosError(axiosError, 'Failed to create supplier');
    return NextResponse.json({ error: errorResponse.error }, { status: errorResponse.status });
  }
}

// PUT update supplier
export async function PUT(request: Request) {
  try {
    const headers = getAuthHeaders(request);
    if (!headers.Authorization) {
      const errorResponse = showAxiosError(null, 'Authentication token missing');
      return NextResponse.json({ error: errorResponse.error }, { status: 401 });
    }

    const body = await request.json();
    const { supplier_id, ...updateData } = body;

    if (!supplier_id) {
      return NextResponse.json({ error: 'Supplier ID is required' }, { status: 400 });
    }

    const response: AxiosResponse<SupplierResponse> = await axios.put(
      `${API_BASE_URL}/update-supplier/${supplier_id}`,
      toFormData(updateData),
      {
        headers: {
          ...headers,
          'Content-Type': 'application/x-www-form-urlencoded',
        },
      }
    );

    return NextResponse.json(response.data);
  } catch (error) {
    const axiosError = error as AxiosError<unknown>;
    const errorResponse = showAxiosError(axiosError, 'Failed to update supplier');
    return NextResponse.json({ error: errorResponse.error }, { status: errorResponse.status });
  }
}

// DELETE supplier
export async function DELETE(request: Request) {
  try {
    const headers = getAuthHeaders(request);
    if (!headers.Authorization) {
      const errorResponse = showAxiosError(null, 'Authentication token missing');
      return NextResponse.json({ error: errorResponse.error }, { status: 401 });
    }

    const { searchParams } = new URL(request.url);
    const supplier_id = searchParams.get('supplier_id');

    if (!supplier_id) {
      return NextResponse.json({ error: 'Supplier ID is required' }, { status: 400 });
    }

    const response: AxiosResponse<SupplierResponse> = await axios.delete(
      `${API_BASE_URL}/delete-supplier/${supplier_id}`,
      { headers }
    );

    return NextResponse.json(response.data);
  } catch (error) {
    const axiosError = error as AxiosError<unknown>;
    const errorResponse = showAxiosError(axiosError, 'Failed to delete supplier');
    return NextResponse.json({ error: errorResponse.error }, { status: errorResponse.status });
  }
}