"use client";

import {
  createContext,
  useContext,
  useState,
  useEffect,
  ReactNode,
} from "react";
import { useRouter } from "next/navigation";
import { API_BASE_PATH } from "@/config/apiConfig";

interface User {
  user_id: string;
  name: string;
  role: string;
  // email: string; // Added email field
}

interface LoginResponse {
  user_id: string;
  name: string;
  role: string;
  // email: string; // Added email field to match the response
  access_token: string;
  token_type: string;
  expires_in: number;
}

interface AuthContextType {
  user: User | null;
  loading: boolean;
  login: (
    username: string,
    password: string,
    remember_me: boolean
  ) => Promise<boolean>;
  logout: () => void;
}

const AuthContext = createContext<AuthContextType | undefined>(undefined);

export const AuthProvider = ({ children }: { children: ReactNode }) => {
  const [user, setUser] = useState<User | null>(null);
  const [loading, setLoading] = useState(true);
  const router = useRouter(); // ✅ Router hook

  // Restore user from localStorage
  useEffect(() => {
    const storedUser = localStorage.getItem("userData");
    if (storedUser) {
      setUser(JSON.parse(storedUser));
    }
    setLoading(false);
  }, []);

  const login = async (
    username: string,
    password: string,
    remember_me: boolean
  ): Promise<boolean> => {
    try {
      const response = await fetch("/api/login", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          username,
          password,
          remember_me: remember_me ? "true" : "false",
        }),
      });

      if (!response.ok) {
        const errData = await response.json();
        throw new Error(errData.detail || errData.message || "Login failed");
      }

      const data: LoginResponse = await response.json();
      const userData: User = {
        user_id: data.user_id,
        name: data.name,
        role: data.role,
        // email: data.email, // Added email from response
      };

      setUser(userData);
      const fullToken = `${data.token_type} ${data.access_token}`;
      localStorage.setItem("authToken", fullToken);
      localStorage.setItem("userData", JSON.stringify(userData));
      localStorage.setItem("authExpiry", String(Date.now() + data.expires_in * 1000));

      const role = data.role.toLowerCase();
      if (role === "hr") router.push("/hrm");
      else router.push("/dashboard");

      return true;
    } catch (err) {
      console.error("Login failed:", err);
      return false;
    }
  };

  const logout = async () => {
    try {
      const token = localStorage.getItem("authToken");
      if (token) {
        await fetch(`${API_BASE_PATH}/logout`, {
          method: "POST",
          headers: {
            Authorization: token,
            "Content-Type": "application/json",
          },
        });
      }
    } catch (error) {
      console.error("Logout API error:", error);
    } finally {
      // ✅ always clear session on frontend
      setUser(null);
      localStorage.removeItem("authToken");
      localStorage.removeItem("userData");
      localStorage.removeItem("authExpiry");
      router.push("/login"); // ✅ Redirect after logout
    }
  };

  return (
    <AuthContext.Provider value={{ user, loading, login, logout }}>
      {children}
    </AuthContext.Provider>
  );
};

export const useAuth = (): AuthContextType => {
  const context = useContext(AuthContext);
  if (context === undefined)
    throw new Error("useAuth must be used within an AuthProvider");
  return context;
};