"use client";

import React, { useCallback, useEffect, useMemo, useState } from "react";
import Button from "antd/es/button";
import "antd/es/button/style";
import Card from "antd/es/card";
import "antd/es/card/style";
import DatePicker from "antd/es/date-picker";
import "antd/es/date-picker/style";
import Form from "antd/es/form";
import "antd/es/form/style";
import Input from "antd/es/input";
import "antd/es/input/style";
import Modal from "antd/es/modal";
import "antd/es/modal/style";
import Space from "antd/es/space";
import "antd/es/space/style";
import Switch from "antd/es/switch";
import "antd/es/switch/style";
import Tooltip from "antd/es/tooltip";
import "antd/es/tooltip/style";
import Tag from "antd/es/tag";
import "antd/es/tag/style";
import Breadcrumb from "antd/es/breadcrumb";
import "antd/es/breadcrumb/style";
import Pagination from "antd/es/pagination";
import "antd/es/pagination/style";
import Dropdown from "antd/es/dropdown";
import "antd/es/dropdown/style";
import Select from "antd/es/select";
import "antd/es/select/style";
import Upload from "antd/es/upload";
import "antd/es/upload/style";

import {
  PlusOutlined,
  EditOutlined,
  DeleteOutlined,
  EyeOutlined,
  HomeOutlined,
  TableOutlined,
  AppstoreOutlined,
  SearchOutlined,
  DownloadOutlined,
  CheckCircleTwoTone,
  WarningOutlined,
  StopOutlined,
  CheckCircleOutlined,
  CloseCircleOutlined,
  UploadOutlined,
} from "@ant-design/icons";
import type { ColumnsType } from "antd/es/table";
import DataTable from "../components/common/DataTable";
//import WidgetCard from "../components/common/WidgetCard";
//import ColumnVisibilityToggle from "../components/common/ManageColumn";
import { useIsMobile } from "@/data/useIsMobile";
import { toast } from "sonner";
import { motion } from "framer-motion";
import {
  createSupplierTest,
  deleteSupplierTest,
  downloadSupplierTestFile,
  getAllSupplierTests,
  SupplierTest,
  updateSupplierTest,
} from "../api/suppliertest/suppliertest";
import { getAllProducts, Product } from "../api/products/product";
import { getAllSuppliers, Supplier } from "../api/suppliers/suppliers";
//import ConfirmDialog from "../components/common/ConfirmDialog";
import dayjs from "@/data/utils/dayjs";
import dynamic from "next/dynamic";
import GlassCard from "../components/common/GlassCard";
import { FileEdit, TestTube2 } from "lucide-react";
import { FlaskConicalIcon } from "lucide-react";
// ✅ Dynamic import - WidgetCard
const WidgetCard = dynamic(() => import("../components/common/WidgetCard"), {
  ssr: false,
  loading: () => <p className="text-sm text-gray-400">Loading widget...</p>,
});

// ✅ Dynamic import - ColumnVisibilityToggle
const ColumnVisibilityToggle = dynamic(
  () => import("../components/common/ManageColumn"),
  {
    ssr: false,
    loading: () => <p className="text-sm text-gray-400">Loading columns...</p>,
  }
);

// ✅ Dynamic import - ConfirmDialog
const ConfirmDialog = dynamic(
  () => import("../components/common/ConfirmDialog"),
  {
    ssr: false,
    loading: () => <p className="text-sm text-gray-400">Preparing dialog...</p>,
  }
);

const { RangePicker } = DatePicker;
const inspectionResultOptions = ["Passed", "Failed"];
const testMethodOptions = ["Visual", "Lab", "Measurement"];
const statusOptions = ["Draft", "Submitted", "Approved", "Rejected"];

export default function SampleTestPage() {
  const [form] = Form.useForm();
  const [tests, setTests] = useState<SupplierTest[]>([]);
  const [products, setProducts] = useState<Product[]>([]);
  const [suppliers, setSuppliers] = useState<Supplier[]>([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [filteredProducts, setFilteredProducts] = useState<Product[]>([]);
  const [selectedSupplier, setSelectedSupplier] = useState<string | null>(null);
  const [editingTest, setEditingTest] = useState<SupplierTest | null>(null);
  const [viewingTest, setViewingTest] = useState<SupplierTest | null>(null);
  const [searchTerm, setSearchTerm] = useState("");
  const [isGridView, setIsGridView] = useState(false);
  const [selectedRows, setSelectedRows] = useState<string[]>([]);
  const [advancedFilters, setAdvancedFilters] = useState({
    inspectionResult: undefined,
  });
  const [confirmDialog, setConfirmDialog] = useState({
    open: false,
    onConfirm: () => {},
    title: "",
    message: "",
  });

  const [dateRange, setDateRange] = useState<[dayjs.Dayjs, dayjs.Dayjs] | null>(
    null
  );
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(10);
  const isMobile = useIsMobile();

  useEffect(() => {
    const fetchInit = async () => {
      const [prods, sups, testData] = await Promise.all([
        getAllProducts(),
        getAllSuppliers(),
        getAllSupplierTests(),
      ]);
      setProducts(prods);
      setSuppliers(sups);
      setTests(testData);
    };
    fetchInit();
  }, []);

  const openAddModal = () => {
    form.resetFields();
    form.setFieldsValue({
      poNumber: "",
      testMethod: undefined,
      sampleBatchNo: "",
      status: undefined,
      testResultDetails: "",
    });
    setEditingTest(null);
    setIsModalOpen(true);
  };
  const openEditModal = useCallback(
    (test: SupplierTest) => {
      form.setFieldsValue({
        ...test,
        poDate: test.poDate ? dayjs(test.poDate) : undefined,
        deliveryDate: test.deliveryDate ? dayjs(test.deliveryDate) : undefined,
        receivingDate: test.receivingDate
          ? dayjs(test.receivingDate)
          : undefined,
        testDate: test.testDate ? dayjs(test.testDate) : undefined,
        testReportFile: test.testReportFile
          ? [
              {
                uid: "-1",
                name:
                  typeof test.testReportFile === "string"
                    ? (test.testReportFile as string).split("/").pop()
                    : "existing_file.pdf",
                status: "done",
              },
            ]
          : [],
      });
      setSelectedSupplier(test.supplier_id);
      setEditingTest(test);
      setIsModalOpen(true);
    },
    [form, setSelectedSupplier, setEditingTest, setIsModalOpen]
  );

  const handleDelete = useCallback(
    (supplierId: string, productId: string) => {
      setConfirmDialog({
        open: true,
        title: "Delete Test",
        message: "Are you sure you want to delete this test?",
        onConfirm: async () => {
          try {
            await deleteSupplierTest(supplierId, productId);
            const refreshed = await getAllSupplierTests();
            setTests(refreshed);
            toast.success("Test deleted successfully");
          } catch {
            toast.error("Failed to delete test");
          }
        },
      });
    },
    [setTests, setConfirmDialog]
  );
  const handleBulkDelete = useCallback(() => {
    if (selectedRows.length === 0) return;
    setConfirmDialog({
      open: true,
      title: "Delete Multiple Tests",
      message: `Are you sure you want to delete ${selectedRows.length} tests?`,
      onConfirm: async () => {
        try {
          await Promise.all(
            selectedRows.map((rowId) => {
              const test = tests.find((t) => t.supplier_id === rowId);
              return test
                ? deleteSupplierTest(test.supplier_id, test.product_id)
                : Promise.resolve();
            })
          );
          const refreshed = await getAllSupplierTests();
          setTests(refreshed);
          setSelectedRows([]);
          toast.success("Tests deleted successfully");
        } catch {
          toast.error("Failed to delete selected tests");
        }
      },
    });
  }, [selectedRows, tests, setTests, setSelectedRows, setConfirmDialog]);

  const handleSupplierChange = useCallback(
    (supplierId: string) => {
      setSelectedSupplier(supplierId);
      form.setFieldsValue({ product_id: undefined });
      const filtered = products.filter((p) => p.supplier_id === supplierId);
      setFilteredProducts(filtered);
    },
    [form, products, setSelectedSupplier, setFilteredProducts]
  );

  const handleFormSubmit = async () => {
    try {
      // const fields = [
      //   "supplier_id",
      //   "product_id",
      //   "poNumber",
      //   "poDate",
      //   "deliveryDate",
      //   "receivingDate",
      //   "testDate",
      //   "testedBy",
      //   "inspectionResult",
      //   "testMethod",
      //   "sampleBatchNo",
      //   "status",
      //   "testResultDetails",
      //   "rejectionReason",
      //   "testReportFile"
      // ];
      // for (const field of fields) {
      //   try {
      //     await form.validateFields([field]);
      //     console.log(`Field ${field} validated successfully`);
      //   } catch (error) {
      //     console.error(`Validation failed for ${field}:`, error);
      //   }
      // }
      // console.log("Form Values Before Validation:", form.getFieldsValue());
      const values = await form.validateFields();
      // console.log("Form Values After Validation:", values);

      const exists = tests.some(
        (t) =>
          t.supplier_id === values.supplier_id &&
          t.product_id === values.product_id &&
          (!editingTest ||
            editingTest.product_id !== t.product_id ||
            editingTest.supplier_id !== t.supplier_id)
      );

      if (!editingTest && exists) {
        toast.warning("Test for this supplier and product already exists.");
        return;
      }
      const payload = {
        supplier_id: values.supplier_id,
        product_id: values.product_id,
        poNumber: values.poNumber,
        poDate: values.poDate ? dayjs(values.poDate).format("YYYY-MM-DD") : "",
        deliveryDate: values.deliveryDate
          ? dayjs(values.deliveryDate).format("YYYY-MM-DD")
          : "",
        receivingDate: values.receivingDate
          ? dayjs(values.receivingDate).format("YYYY-MM-DD")
          : "",
        testDate: values.testDate
          ? dayjs(values.testDate).format("YYYY-MM-DD")
          : "",
        testedBy: values.testedBy,
        testMethod: values.testMethod,
        inspectionResult: values.inspectionResult,
        sampleBatchNo: values.sampleBatchNo,
        testResultDetails: values.testResultDetails,
        rejectionReason: values.rejectionReason,
        status: values.status,
        testReportFile:
          Array.isArray(values.testReportFile) &&
          values.testReportFile[0]?.originFileObj
            ? values.testReportFile[0].originFileObj
            : null,
        samplequantity: values.samplequantity || "",
      };

      if (editingTest) {
        await updateSupplierTest(
          editingTest.supplier_id,
          editingTest.product_id,
          payload
        );
        toast.success("Supplier test updated successfully");
      } else {
        await createSupplierTest(payload);
        toast.success("Supplier test created successfully");
      }
      setIsModalOpen(false);
      form.resetFields();
      const refreshed = await getAllSupplierTests();
      setTests(refreshed);
    } catch (error) {
      console.error(error);
      toast.error("Failed to submit supplier test");
    }
  };

  const renderTag = useCallback((value: string) => {
    const colorMap: Record<string, string> = {
      Pass: "green",
      Fail: "red",
      Pending: "orange",
    };

    const iconMap: Record<string, React.ReactNode> = {
      Pass: <CheckCircleOutlined />,
      Fail: <CloseCircleOutlined />,
      Pending: <WarningOutlined />,
    };

    return (
      <Tag icon={iconMap[value]} color={colorMap[value] || "default"}>
        {value}
      </Tag>
    );
  }, []); // No dependencies since colorMap and iconMap are static

  const filteredTests = useMemo(() => {
    return tests.filter((test) => {
      const matchesSearch =
        test.supplier_name!.toLowerCase().includes(searchTerm.toLowerCase()) ||
        test.product_name!.toLowerCase().includes(searchTerm.toLowerCase()) ||
        test
          .created_by_name!.toLowerCase()
          .includes(searchTerm.toLowerCase()) ||
        test.testedBy.toLowerCase().includes(searchTerm.toLowerCase());

      const matchesDate =
        !dateRange ||
        (test.created_at &&
          (!dateRange[0] ||
            dayjs(test.created_at).isSame(dateRange[0]) ||
            dayjs(test.created_at).isAfter(dateRange[0])) &&
          (!dateRange[1] ||
            dayjs(test.created_at).isSame(dateRange[1]) ||
            dayjs(test.created_at).isBefore(dateRange[1])));
      const matchesAdvanced =
        !advancedFilters.inspectionResult ||
        test.inspectionResult === advancedFilters.inspectionResult;
      return matchesSearch && matchesDate && matchesAdvanced;
    });
  }, [tests, searchTerm, dateRange, advancedFilters]);

  const paginatedTests = useMemo(() => {
    return filteredTests.slice(
      (currentPage - 1) * pageSize,
      currentPage * pageSize
    );
  }, [filteredTests, currentPage, pageSize]);

  const handleDownload = useCallback(async (test: SupplierTest) => {
    try {
      const res = await downloadSupplierTestFile(
        test.supplier_id,
        test.product_id
      );
      const fileURL = window.URL.createObjectURL(new Blob([res.data]));
      const link = document.createElement("a");
      link.href = fileURL;
      link.setAttribute("download", "SupplierTestReport.pdf");
      document.body.appendChild(link);
      link.click();
      link.remove();
    } catch (err) {
      toast.error("Failed to download report file");
      console.error(err);
    }
  }, []);

  const columns: ColumnsType<SupplierTest> = useMemo(
    () => [
      {
        title: (
          <input
            type="checkbox"
            onChange={(e) => {
              const checked = e.target.checked;
              setSelectedRows(
                checked
                  ? filteredTests.map((t) => t.supplier_id.toString())
                  : []
              );
            }}
            className="w-4 h-4"
          />
        ),
        key: "selection",
        render: (_, record) => (
          <input
            type="checkbox"
            checked={selectedRows.includes(record.supplier_id.toString())}
            onChange={(e) => {
              setSelectedRows((prev) =>
                e.target.checked
                  ? [...prev, record.supplier_id.toString()]
                  : prev.filter((id) => id !== record.supplier_id.toString())
              );
            }}
            className="w-4 h-4"
          />
        ),
        width: 44,
      },
      {
        title: "Supplier",
        dataIndex: "supplier_name",
        key: "supplier",
        sorter: (a, b) => a.supplier_name!.localeCompare(b.supplier_name!),
        render: (text) => (
          <span className="text-green-600 font-medium">{text}</span>
        ),
        filterSearch: true,
        onFilter: (value, record) =>
          record
            .supplier_name!.toLowerCase()
            .includes((value as string).toLowerCase()),
        filters: Array.from(new Set(tests.map((t) => t.supplier_name)))
          .filter(
            (value): value is string =>
              typeof value === "string" && value !== undefined
          )
          .map((value) => ({
            text: value,
            value: value,
          })),
      },
      {
        title: "Product",
        dataIndex: "product_name",
        key: "Product",
        sorter: (a, b) => a.product_name!.localeCompare(b.product_name!),
        filterSearch: true,
        onFilter: (value, record) =>
          record
            .product_name!.toLowerCase()
            .includes((value as string).toLowerCase()),
        filters: Array.from(new Set(tests.map((t) => t.product_name)))
          .filter(
            (value): value is string =>
              typeof value === "string" && value !== undefined
          )
          .map((value) => ({
            text: value,
            value: value,
          })),
      },
      {
        title: "Purchase Order No.",
        dataIndex: "poNumber",
        key: "poNumber",
        sorter: (a, b) => a.poNumber.localeCompare(b.poNumber),
      },
      {
        title: "Purchase Order Date",
        dataIndex: "poDate",
        key: "poDate",
        sorter: (a, b) =>
          new Date(a.poDate ?? 0).getTime() - new Date(b.poDate ?? 0).getTime(),
        render: (value: string) =>
          value
            ? `${dayjs(value).format("DD-MM-YYYY hh:mm A")} (${dayjs(
                value
              ).fromNow()})`
            : "-",
      },
      {
        title: "Delivery Date",
        dataIndex: "deliveryDate",
        key: "deliveryDate",
        sorter: (a, b) =>
          new Date(a.deliveryDate ?? 0).getTime() -
          new Date(b.deliveryDate ?? 0).getTime(),
        render: (value: string) =>
          value
            ? `${dayjs(value).format("DD-MM-YYYY hh:mm A")} (${dayjs(
                value
              ).fromNow()})`
            : "-",
      },
      {
        title: "Receiving Date",
        dataIndex: "receivingDate",
        key: "receivingDate",
        sorter: (a, b) =>
          new Date(a.receivingDate ?? 0).getTime() -
          new Date(b.receivingDate ?? 0).getTime(),
        render: (value: string) =>
          value
            ? `${dayjs(value).format("DD-MM-YYYY hh:mm A")} (${dayjs(
                value
              ).fromNow()})`
            : "-",
      },
      {
        title: "Inspection Result",
        dataIndex: "inspectionResult",
        key: "inspectionResult",
        sorter: (a, b) => a.inspectionResult.localeCompare(b.inspectionResult),
        filterSearch: true,
        onFilter: (value, record) => record.inspectionResult === value,
        filters: inspectionResultOptions.map((res) => ({
          text: res,
          value: res,
        })),
        render: renderTag,
      },
      {
        title: "Rejection Reason",
        dataIndex: "rejectionReason",
        key: "rejectionReason",
        sorter: (a, b) =>
          (a.rejectionReason || "").localeCompare(b.rejectionReason || ""),
        render: (text) => (
          <Tooltip title={text}>
            <span className="block truncate max-w-[100px] sm:max-w-[150px] md:max-w-[180px]">
              {text || "-"}
            </span>
          </Tooltip>
        ),
      },
      {
        title: "Test Information",
        key: "testInfo",
        children: [
          {
            title: "Tested By",
            dataIndex: "testedBy",
            key: "testedBy",
            render: (text) => text || "-",
          },
          {
            title: "Test Date",
            dataIndex: "testDate",
            key: "testDate",
            render: (value) =>
              value
                ? `${dayjs(value).format("DD-MM-YYYY hh:mm A")} (${dayjs(
                    value
                  ).fromNow()})`
                : "-",
          },
          {
            title: "Test Method",
            dataIndex: "testMethod",
            key: "testMethod",
            render: (text) => text || "-",
          },
        ],
      },
      {
        title: "Sample Info",
        key: "sampleInfo",
        children: [
          {
            title: "Sample Batch No",
            dataIndex: "sampleBatchNo",
            key: "sampleBatchNo",
            render: (text) => text || "-",
          },
          {
            title: "Sample Qty",
            dataIndex: "samplequantity",
            key: "samplequantity",
            render: (text) => text || "-",
          },

          {
            title: "Test Result Details",
            dataIndex: "testResultDetails",
            key: "testResultDetails",
            render: (text) => (
              <Tooltip title={text}>
                <span className="block truncate max-w-[160px]">
                  {text || "-"}
                </span>
              </Tooltip>
            ),
          },
        ],
      },
      {
        title: "Status",
        dataIndex: "status",
        key: "status",
        render: renderTag,
      },
      {
        title: "Test Report File",
        dataIndex: "testReportFile",
        key: "testReportFile",
        render: (_, record) =>
          record.testReportFile ? (
            <Button
              icon={<DownloadOutlined />}
              size="small"
              onClick={async () => {
                try {
                  const response = await downloadSupplierTestFile(
                    record.supplier_id,
                    record.product_id
                  );
                  const blob = new Blob([response.data]);
                  const url = window.URL.createObjectURL(blob);

                  const link = document.createElement("a");
                  link.href = url;

                  // Determine file name and extension
                  let fileName = "SupplierTest";
                  let fileExt = "pdf";

                  if (typeof record.testReportFile === "string") {
                    const parts = (record.testReportFile as string).split(".");
                    fileExt = parts.length > 1 ? parts.pop()! : "pdf";
                    fileName = parts.join(".");
                  } else if (record.testReportFile instanceof File) {
                    const parts = record.testReportFile.name.split(".");
                    fileExt = parts.length > 1 ? parts.pop()! : "pdf";
                    fileName = parts.join(".");
                  }

                  link.download = `${fileName}.${fileExt}`;

                  document.body.appendChild(link);
                  link.click();
                  document.body.removeChild(link);
                  window.URL.revokeObjectURL(url);
                } catch (error) {
                  toast.error("Failed to download test report");
                  console.error("Download error:", error);
                }
              }}
            >
              Download
            </Button>
          ) : (
            "-"
          ),
      },
      {
        title: "Created At",
        dataIndex: "created_at",
        key: "created_at",
        render: (value) =>
          value
            ? `${dayjs(value).format("DD-MM-YYYY hh:mm A")} (${dayjs(
                value
              ).fromNow()})`
            : "-",
      },
      {
        title: "Updated At",
        dataIndex: "updated_at",
        key: "updated_at",
        render: (value) =>
          value
            ? `${dayjs(value).format("DD-MM-YYYY hh:mm A")} (${dayjs(
                value
              ).fromNow()})`
            : "-",
      },
      {
        title: "Created By",
        dataIndex: "created_by_name",
        key: "created_by_name",
        render: (text) => text || "-",
      },
      {
        title: "Updated By",
        dataIndex: "updated_by_name",
        key: "updated_by_name",
        render: (text) => text || "-",
      },

      {
        title: "Actions",
        key: "actions",
        fixed: "right",
        render: (_, record) => (
          <Space size="small">
            <Tooltip title="View">
              <Button
                icon={<EyeOutlined />}
                onClick={() => setViewingTest(record)}
                size="small"
                className="text-blue-600"
              />
            </Tooltip>
            <Tooltip title="Edit">
              <Button
                icon={<EditOutlined />}
                onClick={() => openEditModal(record)}
                size="small"
                type="primary"
                className="bg-blue-600 hover:bg-blue-700"
              />
            </Tooltip>
            <Tooltip title="Delete">
              <Button
                icon={<DeleteOutlined />}
                onClick={() =>
                  handleDelete(record.supplier_id, record.product_id)
                }
                danger
                size="small"
                className="text-red-600"
              />
            </Tooltip>
          </Space>
        ),
      },
    ],
    [filteredTests, tests, openEditModal, selectedRows, handleDelete, renderTag]
  );

  const allColumns = [
    { key: "selection", label: "Select" },
    { key: "supplier", label: "Supplier" },
    { key: "Product", label: "Product" },
    { key: "poNumber", label: "PO Number" },
    { key: "poDate", label: "PO Date" },
    { key: "deliveryDate", label: "Delivery Date" },
    { key: "receivingDate", label: "Receiving Date" },
    { key: "testInfo", label: "Test Info" },
    { key: "testedBy", label: "Tested By" },
    { key: "testDate", label: "Test Date" },
    { key: "testMethod", label: "Test Method" },
    { key: "sampleInfo", label: "Sample Info" },
    { key: "sampleBatchNo", label: "Sample Batch No" },
    { key: "samplequantity", label: "Sample Quantity" },
    { key: "testResultDetails", label: "Test Result Details" },
    { key: "inspectionResult", label: "Inspection Result" },
    { key: "status", label: "Status" },
    { key: "rejectionReason", label: "Rejection Reason" },
    { key: "testReportFile", label: "Test Report File" },
    { key: "created_at", label: "Created At" },
    { key: "updated_at", label: "Updated At" },
    { key: "created_by_name", label: "Created By" },
    { key: "updated_by_name", label: "Updated By" },
    { key: "actions", label: "Actions" },
  ];

  const columnKeys = allColumns.map((c) => c.key);
  const labelsMap = Object.fromEntries(allColumns.map((c) => [c.key, c.label]));
  const [visibleColumnKeys, setVisibleColumnKeys] =
    useState<string[]>(columnKeys);

  const handleVisibleColumnChange = useCallback((cols: string[]) => {
    setVisibleColumnKeys(cols);
  }, []);

  const filteredColumns = useMemo(() => {
    return columns.filter(
      (col) =>
        typeof col.key === "string" && visibleColumnKeys.includes(col.key)
    );
  }, [columns, visibleColumnKeys]);

  const SampleCard = React.memo(({ test }: { test: SupplierTest }) => (
    <GlassCard gradient blurStrength="lg" hoverLift color="success">
      <div
        className="border rounded-xl p-4 shadow-sm hover:shadow-md bg-white dark:bg-gray-900 
                    transition-all w-full flex flex-col"
      >
        {/* Header */}
        <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-4">
          <h3 className="text-lg font-semibold text-green-600 dark:text-white truncate">
            {test.supplier_name || "N/A"}
          </h3>
          <Tag color="blue">{test.inspectionResult || "N/A"}</Tag>
        </div>

        {/* Line by Line Info */}
        <div className="text-sm space-y-2">
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Item:</strong> {test.product_name || "N/A"}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>PO Number:</strong> {test.poNumber}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>PO Date:</strong> {dayjs(test.poDate).format("YYYY-MM-DD")}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Delivery Date:</strong>{" "}
            {dayjs(test.deliveryDate).format("YYYY-MM-DD")}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Received:</strong>{" "}
            {dayjs(test.receivingDate).format("YYYY-MM-DD")}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Test Date:</strong>{" "}
            {dayjs(test.testDate).format("YYYY-MM-DD")}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Tested By:</strong> {test.testedBy || "N/A"}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Test Method:</strong> {test.testMethod || "N/A"}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Status:</strong> {renderTag(test.status)}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Result:</strong> {renderTag(test.inspectionResult)}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Sample Quantity:</strong> {test.samplequantity || "N/A"}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Rejection Reason:</strong> {test.rejectionReason || "N/A"}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Batch No:</strong> {test.sampleBatchNo || "N/A"}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Test Result Details:</strong>{" "}
            {test.testResultDetails || "N/A"}
          </p>
          <p className="text-gray-600 dark:text-gray-300">
            <strong>Test Report File:</strong>{" "}
            {test.testReportFile
              ? typeof test.testReportFile === "string"
                ? test.testReportFile
                : test.testReportFile.name
              : "N/A"}
          </p>
        </div>

        {/* Download Report */}
        {test.testReportFile && (
          <Button
            type="link"
            size="small"
            onClick={() => handleDownload(test)}
            icon={<DownloadOutlined />}
            className="pl-0 mt-2"
          >
            Download Report
          </Button>
        )}

        {/* Footer Actions */}
        <div className="mt-3 flex gap-2">
          <Tooltip title="View">
            <Button
              icon={<EyeOutlined />}
              onClick={() => setViewingTest(test)}
              size="small"
              className="text-blue-600"
            />
          </Tooltip>
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => openEditModal(test)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => handleDelete(test.supplier_id, test.product_id)}
              size="small"
              danger
              className="text-red-600"
            />
          </Tooltip>
        </div>
      </div>
    </GlassCard>
  ));
  SampleCard.displayName = "SampleCard";

  const total = tests.length;
  const passed = tests.filter((t) => t.inspectionResult === "Pass").length;
  const failed = tests.filter((t) => t.inspectionResult === "Fail").length;
  const pending = tests.filter((t) => t.inspectionResult === "Pending").length;

  const trendData = [
    { value: 10 },
    { value: 15 },
    { value: 12 },
    { value: 18 },
    { value: 20 },
  ];

  return (
    <>
      <div className="px-2 sm:px-2 md:px-2 lg:px-2 md:py-2 lg:py-2 py-2 sm:py-2">
        <h2 className="flex items-center gap-2 text-xl sm:text-2xl md:text-3xl font-bold text-gray-800 dark:text-white mb-2">
          <FlaskConicalIcon className="w-7 h-7 text-green-600 dark:text-green-400 flex-shrink-0" />
          <span>Sample Tests</span>
          <span className="text-gray-500 text-base sm:text-lg font-medium">
            ({tests.length})
          </span>
        </h2>
        <p className="text-gray-600 dark:text-gray-300 text-sm sm:text-base mb-4">
          Track sample inspection records for deliveries.
        </p>
        <Breadcrumb
          className="mb-4 sm:mb-6 mt-2 text-sm text-gray-500 dark:text-gray-400"
          separator=">"
          items={[
            {
              title: (
                <Tooltip title="Dashboard">
                  <HomeOutlined />
                </Tooltip>
              ),
              href: "/dashboard",
            },
            {
              title: "Master Data",
            },
            {
              title: "Sample Tests",
            },
          ]}
        />

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mt-3 mb-6">
          <WidgetCard
            title="Total Tests"
            value={total}
            icon={<AppstoreOutlined />}
            color="green"
            percentage={12}
            description="All sample tests conducted"
            trendData={trendData}
          />
          <WidgetCard
            title="Passed Tests"
            value={passed}
            icon={<CheckCircleTwoTone twoToneColor="#52c41a" />}
            color="blue"
            percentage={8}
            description="Tests with pass result"
            trendData={trendData}
          />
          <WidgetCard
            title="Failed Tests"
            value={failed}
            icon={<StopOutlined />}
            color="red"
            percentage={-3}
            description="Tests with fail result"
            trendData={trendData}
          />
          <WidgetCard
            title="Pending Tests"
            value={pending}
            icon={<WarningOutlined />}
            color="orange"
            percentage={5}
            description="Tests awaiting results"
            trendData={trendData}
          />
        </div>

        <div className="mt-4">
          <Card title="Sample Test List" className="p-0 mb-6 shadow-sm">
            <div className="p-4 flex flex-col gap-4">
              <div className="flex flex-col sm:flex-row justify-between items-stretch sm:items-center gap-4">
                <Input
                  prefix={<SearchOutlined />}
                  allowClear
                  placeholder="Search Supplier or Product"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="w-full sm:w-64 border-gray-300 dark:border-gray-600 rounded-md shadow-sm"
                />
                <Dropdown
                  className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  menu={{
                    items: [
                      { label: "Export CSV", key: "csv" },
                      { label: "Export PDF", key: "pdf" },
                    ],
                  }}
                >
                  <Button icon={<DownloadOutlined />}>Export</Button>
                </Dropdown>
                <Button
                  type="primary"
                  icon={<PlusOutlined />}
                  onClick={openAddModal}
                  className="bg-blue-600 hover:bg-blue-700 w-full sm:w-auto"
                >
                  Add Sample Test
                </Button>
              </div>
              <div className="flex flex-col gap-4 lg:flex-row lg:justify-between lg:items-center">
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 w-full lg:w-auto">
                  <Select
                    placeholder="Inspection Result"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({
                        ...prev,
                        inspectionResult: value,
                      }))
                    }
                    allowClear
                    options={inspectionResultOptions.map((v) => ({
                      label: v,
                      value: v,
                    }))}
                  />
                </div>
                <div className="flex flex-col sm:flex-row flex-wrap gap-4 justify-start lg:justify-end w-full lg:w-auto">
                  <ColumnVisibilityToggle
                    columns={columnKeys}
                    labels={labelsMap}
                    storageKey="sample_test_table_visibility"
                    onChange={handleVisibleColumnChange}
                  />
                  <RangePicker
                    value={dateRange}
                    onChange={(dates) =>
                      setDateRange(dates as [dayjs.Dayjs, dayjs.Dayjs] | null)
                    }
                    className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  />
                  <Button
                    onClick={handleBulkDelete}
                    disabled={selectedRows.length === 0}
                    danger
                    className="w-full sm:w-auto"
                  >
                    Bulk Delete
                  </Button>
                  <div className="hidden sm:block">
                    <Switch
                      checkedChildren={<AppstoreOutlined />}
                      unCheckedChildren={<TableOutlined />}
                      checked={isGridView}
                      onChange={setIsGridView}
                      className="w-full sm:w-auto"
                    />
                  </div>
                </div>
              </div>

              {advancedFilters.inspectionResult && (
                <div className="flex flex-wrap gap-2 mt-2">
                  <Tag
                    closable
                    onClose={() =>
                      setAdvancedFilters((prev) => ({
                        ...prev,
                        inspectionResult: undefined,
                      }))
                    }
                  >
                    Inspection Result: {advancedFilters.inspectionResult}
                  </Tag>
                </div>
              )}
            </div>

            {isGridView || isMobile ? (
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-3 gap-4 p-4">
                {paginatedTests.map((test) => (
                  <SampleCard key={test.supplier_id} test={test} />
                ))}
              </div>
            ) : (
              <div className="overflow-auto">
                <DataTable<SupplierTest>
                  columns={filteredColumns}
                  dataSource={paginatedTests}
                  rowKey={(record) =>
                    `${record.supplier_id}-${record.product_id}`
                  }
                  scrollX="max-content"
                  className="mt-4"
                  pagination={false}
                  //loading={loading}
                />
              </div>
            )}
            <div className="flex flex-col sm:flex-row justify-between items-center px-4 py-2 text-sm text-gray-600 dark:text-gray-300">
              <div className="mb-2 sm:mb-0">
                Showing <strong>{(currentPage - 1) * pageSize + 1}</strong> to{" "}
                <strong>
                  {Math.min(currentPage * pageSize, filteredTests.length)}
                </strong>{" "}
                of <strong>{filteredTests.length}</strong> tests
              </div>
              <Pagination
                current={currentPage}
                pageSize={pageSize}
                total={filteredTests.length}
                onChange={(page, size) => {
                  setCurrentPage(page);
                  setPageSize(size);
                }}
                showSizeChanger
                pageSizeOptions={["10", "30", "50", "100"]}
                size="small"
              />
            </div>
          </Card>
        </div>

        <motion.div
          initial={{ opacity: 0, scale: 0.95 }}
          animate={{ opacity: 1, scale: 1 }}
          exit={{ opacity: 0, scale: 0.95 }}
        >
          <Modal
            title={
              <div className="flex items-center gap-2 border-b pb-2 border-gray-500 dark:border-gray-600">
                {/* Icon changes based on editing state */}
                {editingTest ? (
                  <FileEdit className="w-5 h-5 text-blue-600" /> // Edit
                ) : (
                  <TestTube2 className="w-5 h-5 text-green-600" /> // Add
                )}
                <span>
                  {editingTest ? "Edit Sample Test" : "Add Sample Test"}
                </span>
              </div>
            }
            open={isModalOpen}
            onOk={handleFormSubmit}
            onCancel={() => {
              setIsModalOpen(false);
              form.resetFields();
            }}
            destroyOnHidden
            width="95%" // responsive width
            className="!top-5 font-semibold"
            styles={{
              body: { maxHeight: "80vh", overflowY: "auto" }, // ✅ fixed line (was bodyStyle)
            }}
          >
            <Form form={form} layout="vertical" className="w-full px-4">
              {/* Supplier & Product Info */}
              <div className="mb-6">
                <h2 className="text-lg font-semibold mb-3 pb-1">
                  Supplier & Product Info
                </h2>
                <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                  <Form.Item
                    name="supplier_id"
                    label="Supplier"
                    rules={[
                      { required: true, message: "Supplier is required" },
                    ]}
                  >
                    <Select
                      showSearch
                      optionFilterProp="label"
                      onChange={handleSupplierChange}
                      options={suppliers.map((s) => ({
                        label: `${s.company_name}${
                          s.contact_person ? ` - ${s.contact_person}` : ""
                        } (${s.supplier_id})`,
                        value: s.supplier_id,
                      }))}
                    />
                  </Form.Item>

                  <Form.Item
                    name="product_id"
                    label="Product"
                    rules={[
                      { required: true, message: "Please select a product" },
                    ]}
                  >
                    <Select
                      showSearch
                      optionFilterProp="label"
                      placeholder={
                        selectedSupplier
                          ? "Select a product"
                          : "Select a supplier first"
                      }
                      options={
                        selectedSupplier
                          ? filteredProducts.map((p) => ({
                              label: `${p.product_name} (${p.product_code})`,
                              value: p.product_id,
                            }))
                          : []
                      }
                    />
                  </Form.Item>
                </div>
              </div>

              {/* Purchase Order Details */}
              <div className="mb-6">
                <h2 className="text-lg font-semibold mb-3 pb-1">
                  Purchase Order Details
                </h2>
                <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                  <Form.Item
                    name="poNumber"
                    label="Purchase Order Number"
                    rules={[{ required: true }]}
                  >
                    <Input placeholder="Enter PO Number" />
                  </Form.Item>

                  <Form.Item
                    name="poDate"
                    label="Purchase Order Date"
                    rules={[{ required: true }]}
                  >
                    <DatePicker className="w-full" />
                  </Form.Item>

                  <Form.Item
                    name="deliveryDate"
                    label="Delivery Date"
                    rules={[{ required: true }]}
                  >
                    <DatePicker className="w-full" />
                  </Form.Item>

                  <Form.Item
                    name="receivingDate"
                    label="Date of Receiving"
                    rules={[{ required: true }]}
                  >
                    <DatePicker className="w-full" />
                  </Form.Item>
                </div>
              </div>

              {/* Testing & Inspection */}
              <div className="mb-6">
                <h2 className="text-lg font-semibold mb-3 pb-1">
                  Testing & Inspection
                </h2>
                <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                  <Form.Item
                    name="testDate"
                    label="Date of Test"
                    rules={[{ required: true }]}
                  >
                    <DatePicker className="w-full" />
                  </Form.Item>

                  <Form.Item
                    name="testedBy"
                    label="Tested By"
                    rules={[{ required: true }]}
                  >
                    <Input placeholder="Person who tested the sample" />
                  </Form.Item>

                  <Form.Item
                    name="inspectionResult"
                    label="Inspection Result"
                    rules={[{ required: true }]}
                  >
                    <Select
                      options={inspectionResultOptions.map((v) => ({
                        label: v,
                        value: v,
                      }))}
                    />
                  </Form.Item>

                  <Form.Item
                    name="testMethod"
                    label="Test Method"
                    rules={[{ required: true }]}
                  >
                    <Select
                      placeholder="Select a test method"
                      options={testMethodOptions.map((v) => ({
                        label: v,
                        value: v,
                      }))}
                    />
                  </Form.Item>

                  <Form.Item
                    name="sampleBatchNo"
                    label="Sample Batch No."
                    rules={[{ required: true }]}
                  >
                    <Input placeholder="Enter Sample Batch Number" />
                  </Form.Item>

                  <Form.Item name="samplequantity" label="Sample Quantity">
                    <Input placeholder="Enter sample quantity" />
                  </Form.Item>
                </div>

                {/* Full width */}
                <div className="col-span-full">
                  <Form.Item
                    name="testResultDetails"
                    label="Test Result Details"
                    rules={[{ required: true }]}
                  >
                    <Input.TextArea
                      rows={3}
                      placeholder="Enter test result summary/details"
                    />
                  </Form.Item>
                </div>

                <div className="col-span-full">
                  <Form.Item
                    name="rejectionReason"
                    label="Reason of Rejection (if any)"
                  >
                    <Input.TextArea rows={2} />
                  </Form.Item>
                </div>
              </div>

              {/* Status & Files */}
              <div className="mb-6">
                <h2 className="text-lg font-semibold mb-3 pb-1">
                  Status & Files
                </h2>
                <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                  <Form.Item
                    name="status"
                    label="Status"
                    rules={[{ required: true }]}
                  >
                    <Select
                      options={statusOptions.map((v) => ({
                        label: v,
                        value: v,
                      }))}
                    />
                  </Form.Item>

                  <div className="col-span-full">
                    <Form.Item
                      name="testReportFile"
                      label="Test Report File"
                      valuePropName="fileList"
                      getValueFromEvent={(e) => {
                        const fileList = Array.isArray(e) ? e : e?.fileList;
                        return Array.isArray(fileList) ? fileList : [];
                      }}
                    >
                      <Upload
                        beforeUpload={() => false}
                        maxCount={1}
                        accept=".pdf,.doc,.docx,.jpg,.png"
                      >
                        <Button icon={<UploadOutlined />}>Select File</Button>
                      </Upload>
                    </Form.Item>
                  </div>
                </div>
              </div>
            </Form>
          </Modal>
        </motion.div>

        <Modal
          title="Sample Test Details"
          open={!!viewingTest}
          onCancel={() => setViewingTest(null)}
          footer={null}
          className="w-full max-w-[90vw] sm:max-w-[600px] top-4"
        >
          {viewingTest && (
            <div className="space-y-2 text-sm sm:text-base">
              <p>
                <strong>Supplier:</strong>{" "}
                <span className="text-green-600">
                  {viewingTest.supplier_name}
                </span>
              </p>
              <p>
                <strong>Product:</strong> {viewingTest.product_name}
              </p>
              <p>
                <strong>PO Number:</strong> {viewingTest.poNumber}
              </p>
              <p>
                <strong>PO Date:</strong>{" "}
                {dayjs(viewingTest.poDate).format("YYYY-MM-DD")}
              </p>
              <p>
                <strong>Delivery Date:</strong>{" "}
                {dayjs(viewingTest.deliveryDate).format("YYYY-MM-DD")}
              </p>
              <p>
                <strong>Date of Receiving:</strong>{" "}
                {dayjs(viewingTest.receivingDate).format("YYYY-MM-DD")}
              </p>

              <p>
                <strong>Test Date:</strong>{" "}
                {dayjs(viewingTest.testDate).format("YYYY-MM-DD")}
              </p>
              <p>
                <strong>Tested By:</strong> {viewingTest.testedBy}
              </p>
              <p>
                <strong>Test Method:</strong> {viewingTest.testMethod}
              </p>
              <p>
                <strong>Sample Batch No.:</strong> {viewingTest.sampleBatchNo}
              </p>

              <p>
                <strong>Sample Quantity:</strong> {viewingTest?.samplequantity}
              </p>

              <p>
                <strong>Inspection Result:</strong>{" "}
                {viewingTest.inspectionResult}
              </p>
              <p>
                <strong>Test Result Details:</strong>{" "}
                {viewingTest.testResultDetails}
              </p>
              <p>
                <strong>Reason of Rejection:</strong>{" "}
                {viewingTest.rejectionReason || "-"}
              </p>
              {viewingTest.testReportFile && (
                <p>
                  <strong>Test Report:</strong>{" "}
                  {viewingTest.testReportFile && (
                    <p>
                      <strong>Test Report:</strong>{" "}
                      <Button
                        type="link"
                        icon={<DownloadOutlined />}
                        onClick={() => handleDownload(viewingTest)}
                      >
                        Download Report File
                      </Button>
                    </p>
                  )}
                </p>
              )}
              <p>
                <strong>Status:</strong> {viewingTest.status}
              </p>
              <p>
                <strong>Test Report File:</strong>{" "}
                {viewingTest.testReportFile instanceof File
                  ? viewingTest.testReportFile.name
                  : "No file"}
              </p>
              <p>
                <strong>Created At:</strong>{" "}
                {dayjs(viewingTest.created_at).format("YYYY-MM-DD")}
              </p>
              <p>
                <strong>Updated At:</strong>{" "}
                {viewingTest.updated_at
                  ? dayjs(viewingTest.updated_at).format("YYYY-MM-DD")
                  : "-"}
              </p>
              <p>
                <strong>Created By:</strong> {viewingTest.created_by_name}
              </p>
              <p>
                <strong>Updated By:</strong>{" "}
                {viewingTest.updated_by_name || "-"}
              </p>
            </div>
          )}
        </Modal>
      </div>
      <ConfirmDialog
        open={confirmDialog.open}
        title={confirmDialog.title}
        message={confirmDialog.message}
        onClose={() => setConfirmDialog((prev) => ({ ...prev, open: false }))}
        onConfirm={async () => {
          await confirmDialog.onConfirm();
          setConfirmDialog((prev) => ({ ...prev, open: false }));
        }}
      />
    </>
  );
}
