"use client";

import React, { useCallback, useState, useEffect, useMemo } from "react";
import Breadcrumb from "antd/es/breadcrumb";
import "antd/es/breadcrumb/style";
import Card from "antd/es/card";
import "antd/es/card/style";
import Input from "antd/es/input";
import "antd/es/input/style";
import Space from "antd/es/space";
import "antd/es/space/style";
import Tooltip from "antd/es/tooltip";
import "antd/es/tooltip/style";
import Tag from "antd/es/tag";
import "antd/es/tag/style";
import Button from "antd/es/button";
import "antd/es/button/style";
import Switch from "antd/es/switch";
import "antd/es/switch/style";
import Modal from "antd/es/modal";
import "antd/es/modal/style";
import Pagination from "antd/es/pagination";
import "antd/es/pagination/style";
import Dropdown from "antd/es/dropdown";
import "antd/es/dropdown/style";
import Select from "antd/es/select";
import "antd/es/select/style";
import {
  HomeOutlined,
  TableOutlined,
  AppstoreOutlined,
  EyeOutlined,
  SearchOutlined,
  DownloadOutlined,
  CheckCircleTwoTone,
  WarningOutlined,
  StopOutlined,
  CheckCircleOutlined,
  PauseCircleOutlined,
  FireOutlined,
  CloseCircleOutlined,
  EditOutlined,
  DeleteOutlined,
} from "@ant-design/icons";
import type { ColumnsType } from "antd/es/table";
import DataTable from "../components/common/DataTable";
import { useIsMobile } from "@/data/useIsMobile";
import { toast } from "sonner";
import { motion } from "framer-motion";
import dynamic from "next/dynamic";
import {
  getAllSupplierApprovals,
  getSupplierApprovalByIds,
  approveSupplierTest,
  downloadSupplierApprovalDocument,
  deleteSupplierApprovalTest,
  SupplierApproval,
} from "../api/supplierapproval/supplierapproval";
//import ConfirmDialog from "../components/common/ConfirmDialog";
import dayjs from "@/data/utils/dayjs";
import GlassCard from "../components/common/GlassCard";
import { ShieldCheckIcon } from "lucide-react";
const WidgetCard = dynamic(() => import("../components/common/WidgetCard"), {
  ssr: false,
});
WidgetCard.displayName = "WidgetCard";

const ColumnVisibilityToggle = dynamic(() => import("../components/common/ManageColumn"), {
  ssr: false,
});
ColumnVisibilityToggle.displayName = "ColumnVisibilityToggle";

const ConfirmDialog = dynamic(() => import("../components/common/ConfirmDialog"), {
  ssr: false,
});
ConfirmDialog.displayName = "ConfirmDialog";

const validationOptions = ["Draft", "Submitted", "Approved", "Rejected"] as const;
const riskOptions = ["Low", "Medium", "High"] as const;

export default function SupplierApprovalPage() {
  const [searchTerm, setSearchTerm] = useState("");
  const [isGridView, setIsGridView] = useState(false);
  const [viewingSupplier, setViewingSupplier] = useState<SupplierApproval | null>(null);
  const [editingSupplier, setEditingSupplier] = useState<SupplierApproval | null>(null);
  const [data, setData] = useState<SupplierApproval[]>([]);
  const [selectedRows, setSelectedRows] = useState<string[]>([]);
  const [advancedFilters, setAdvancedFilters] = useState({
    riskClassification: undefined,
    validation: undefined,
  });
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(10);
  const [loading, setLoading] = useState(false);
  const [confirmDialog, setConfirmDialog] = useState({
    open: false,
    onConfirm: () => { },
    title: "",
    message: "",
  });
  const isMobile = useIsMobile();

  useEffect(() => {
    const fetchData = async () => {
      setLoading(true);
      try {
        const approvals = await getAllSupplierApprovals();
        setData(approvals);
      } catch (error) {
        toast.error("Failed to fetch supplier approvals");
      } finally {
        setLoading(false);
      }
    };
    fetchData();
  }, []);

  const renderTag = useCallback((value: string) => {
    const colorMap: Record<string, string> = {
      Low: "green",
      Medium: "orange",
      High: "red",
      Approved: "green",
      Rejected: "red",
      Pending: "gold",
      Draft: "gray",
      Submitted: "blue",
    };

    const iconMap: Record<string, React.ReactNode> = {
      Low: <CheckCircleOutlined />,
      Medium: <PauseCircleOutlined />,
      High: <FireOutlined />,
      Approved: <CheckCircleOutlined />,
      Rejected: <CloseCircleOutlined />,
      Pending: <WarningOutlined />,
      Draft: <PauseCircleOutlined />,
      Submitted: <CheckCircleOutlined />,
    };

    return (
      <Tag icon={iconMap[value]} color={colorMap[value] || "default"}>
        {value}
      </Tag>
    );
  }, []);

  const filteredData = useMemo(() => {
    return data.filter((supplier) => {
      const matchesSearch =
        (supplier.supplier_name?.toLowerCase().includes(searchTerm.toLowerCase()) || false) ||
        supplier.supplier_id.toLowerCase().includes(searchTerm.toLowerCase()) ||
        (supplier.product_name?.toLowerCase().includes(searchTerm.toLowerCase()) || false);
      const matchesAdvanced =
        (!advancedFilters.riskClassification || supplier.risk_level === advancedFilters.riskClassification) &&
        (!advancedFilters.validation || supplier.status === advancedFilters.validation);
      return matchesSearch && matchesAdvanced;
    });
  }, [data, searchTerm, advancedFilters]);

  const paginatedData = useMemo(() => {
    return filteredData.slice((currentPage - 1) * pageSize, currentPage * pageSize);
  }, [filteredData, currentPage, pageSize]);

  const handleBulkDelete = useCallback(() => {
    if (selectedRows.length === 0) return;
    setConfirmDialog({
      open: true,
      title: "Delete Multiple Supplier Approvals",
      message: `Are you sure you want to delete ${selectedRows.length} supplier approvals?`,
      onConfirm: async () => {
        try {
          await Promise.all(
            selectedRows.map(async (id) => {
              const [supplierId, productId] = id.split("-");
              await deleteSupplierApprovalTest(supplierId, productId);
            })
          );
          const updated = await getAllSupplierApprovals();
          setData(updated);
          setSelectedRows([]);
          toast.success("Supplier approvals deleted successfully");
        } catch {
          toast.error("Failed to delete selected supplier approvals");
        } finally {
          setConfirmDialog((prev) => ({ ...prev, open: false }));
        }
      },
    });
  }, [selectedRows, setData, setSelectedRows, setConfirmDialog]);

  const handleDelete = useCallback((supplierId: string, productId: string) => {
    setConfirmDialog({
      open: true,
      title: "Delete Supplier Approval",
      message: "Are you sure you want to delete this supplier approval?",
      onConfirm: async () => {
        try {
          await deleteSupplierApprovalTest(supplierId, productId);
          const updated = await getAllSupplierApprovals();
          setData(updated);
          toast.success("Supplier approval deleted successfully");
        } catch {
          toast.error("Failed to delete supplier approval");
        } finally {
          setConfirmDialog((prev) => ({ ...prev, open: false }));
        }
      },
    });
  }, [setData, setConfirmDialog]);

  const handleDownload = useCallback(
    async (supplierId: string, productId: string) => {
      try {
        const response = await downloadSupplierApprovalDocument(supplierId, productId);
        const url = window.URL.createObjectURL(new Blob([response.data]));
        const link = document.createElement("a");
        link.href = url;
        link.setAttribute("download", `supplier-approval-${supplierId}-${productId}.pdf`);
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
        toast.success("Document downloaded successfully");
      } catch (error) {
        toast.error("Failed to download document");
      }
    },
    []
  );

  const handleEdit = useCallback(async (supplier: SupplierApproval) => {
    try {
      const supplierData = await getSupplierApprovalByIds(supplier.supplier_id, supplier.product_id);
      setEditingSupplier(supplierData);
    } catch (error) {
      console.error("Failed to fetch supplier details:", error);
      toast.error("Failed to fetch supplier details");
    }
  }, []);

  const handleUpdateStatus = useCallback(async () => {
    if (!editingSupplier) return;
    setIsSubmitting(true);
    try {
      await approveSupplierTest(
        editingSupplier.supplier_id,
        editingSupplier.product_id,
        (editingSupplier.status as "Draft" | "Submitted" | "Approved" | "Rejected") || "Draft"
      );
      setData((prev) =>
        prev.map((s) =>
          s.supplier_id === editingSupplier.supplier_id && s.product_id === editingSupplier.product_id
            ? { ...s, status: editingSupplier.status }
            : s
        )
      );
      setEditingSupplier(null);
      toast.success("Supplier approval status updated successfully");
    } catch (error) {
      console.error("Failed to update supplier approval status:", error);
      toast.error("Failed to update supplier approval status");
    } finally {
      setIsSubmitting(false);
    }
  }, [editingSupplier]);

  const columns: ColumnsType<SupplierApproval> = useMemo(() => [
    {
      title: (
        <input
          type="checkbox"
          onChange={(e) => {
            const checked = e.target.checked;
            setSelectedRows(
              checked ? filteredData.map((s) => `${s.supplier_id}-${s.product_id}`) : []
            );
          }}
          className="w-4 h-4"
        />
      ),
      key: "selection",
      render: (_, record) => (
        <input
          type="checkbox"
          checked={selectedRows.includes(`${record.supplier_id}-${record.product_id}`)}
          onChange={(e) => {
            setSelectedRows((prev) =>
              e.target.checked
                ? [...prev, `${record.supplier_id}-${record.product_id}`]
                : prev.filter((id) => id !== `${record.supplier_id}-${record.product_id}`)
            );
          }}
          className="w-4 h-4"
        />
      ),
      width: 44,
    },
    {
      title: "Supplier ID",
      dataIndex: "supplier_id",
      key: "supplier_id",
      sorter: (a, b) => a.supplier_id.localeCompare(b.supplier_id),
      filterSearch: true,
      onFilter: (value, record) =>
        record.supplier_id.toLowerCase().includes((value as string).toLowerCase()),
      filters: Array.from(new Set(data.map((s) => s.supplier_id))).map((value) => ({
        text: value,
        value: value,
      })),
    },
    {
      title: "Supplier Name",
      dataIndex: "supplier_name",
      key: "supplier_name",
      sorter: (a, b) => (a.supplier_name || "").localeCompare(b.supplier_name || ""),
      render: (text) => <span className="text-green-600 font-medium">{text || "-"}</span>,
      filterSearch: true,
      onFilter: (value, record) =>
        (record.supplier_name || "").toLowerCase().includes((value as string).toLowerCase()),
      filters: Array.from(new Set(data.map((s) => s.supplier_name).filter((v): v is string => !!v))).map((value) => ({
        text: value,
        value: value,
      })),
    },
    {
      title: "Supplier Type",
      dataIndex: "supplier_type",
      key: "supplier_type",
      sorter: (a, b) => (a.supplier_type || "").localeCompare(b.supplier_type || ""),
    },
    {
      title: "Contact Person",
      dataIndex: "contactPerson",
      key: "contactPerson",
      sorter: (a, b) => (a.contactPerson || "").localeCompare(b.contactPerson || ""),
      filterSearch: true,
      onFilter: (value, record) =>
        (record.contactPerson || "").toLowerCase().includes((value as string).toLowerCase()),
      filters: Array.from(new Set(data.map((s) => s.contactPerson).filter((v): v is string => !!v))).map((value) => ({
        text: value,
        value: value,
      })),
    },
    {
      title: "Email",
      dataIndex: "email",
      key: "email",
      sorter: (a, b) => (a.email || "").localeCompare(b.email || ""),
    },
    {
      title: "Phone",
      dataIndex: "phone_number",
      key: "phone_number",
      sorter: (a, b) => (a.phone_number || "").localeCompare(b.phone_number || ""),
    },
    {
      title: "Product ID",
      dataIndex: "product_id",
      key: "product_id",
      sorter: (a, b) => a.product_id.localeCompare(b.product_id),
    },
    {
      title: "Product Name",
      dataIndex: "product_name",
      key: "product_name",
      sorter: (a, b) => (a.product_name || "").localeCompare(b.product_name || ""),
    },
    {
      title: "Product Code",
      dataIndex: "product_code",
      key: "product_code",
      sorter: (a, b) => (a.product_code || "").localeCompare(b.product_code || ""),
    },
    {
      title: "Risk Level",
      dataIndex: "risk_level",
      key: "risk_level",
      filters: riskOptions.map((r) => ({ text: r, value: r })),
      onFilter: (value, record) => record.risk_level === value,
      render: renderTag,
      sorter: (a, b) => (a.risk_level || "").localeCompare(b.risk_level || ""),
    },
    {
      title: "Inspection Result",
      dataIndex: "inspectionResult",
      key: "inspectionResult",
      sorter: (a, b) => (a.inspectionResult || "").localeCompare(b.inspectionResult || ""),
    },
    {
      title: "Tested By",
      dataIndex: "testedBy",
      key: "testedBy",
      sorter: (a, b) => (a.testedBy || "").localeCompare(b.testedBy || ""),
    },
    {
      title: "Test Date",
      dataIndex: "testDate",
      key: "testDate",
      sorter: (a, b) => (a.testDate || "").localeCompare(b.testDate || ""),
    },
    {
      title: "Status",
      dataIndex: "status",
      key: "status",
      filters: validationOptions.map((v) => ({ text: v, value: v })),
      onFilter: (value, record) => record.status === value,
      render: renderTag,
      sorter: (a, b) => (a.status || "").localeCompare(b.status || ""),
    },
    {
      title: "Rejection Reason",
      dataIndex: "rejectionReason",
      key: "rejectionReason",
      sorter: (a, b) => (a.rejectionReason || "").localeCompare(b.rejectionReason || ""),
    },
    {
      title: "Test Result Details",
      dataIndex: "testResultDetails",
      key: "testResultDetails",
      sorter: (a, b) => (a.testResultDetails || "").localeCompare(b.testResultDetails || ""),
    },
    {
      title: "Created At",
      dataIndex: "created_at",
      key: "created_at",
      sorter: (a, b) => (a.created_at || "").localeCompare(b.created_at || ""),
      render: (value) =>
        value ? `${dayjs(value).format("DD-MM-YYYY hh:mm A")} (${dayjs(value).fromNow()})` : "-",
    },
    {
      title: "Approved At",
      dataIndex: "approved_at",
      key: "approved_at",
      render: (value) =>
        value ? `${dayjs(value).format("DD-MM-YYYY hh:mm A")} (${dayjs(value).fromNow()})` : "-",
    },
    {
      title: "Created By",
      dataIndex: "created_by_name",
      key: "created_by_name",
      sorter: (a, b) => (a.created_by_name || "").localeCompare(b.created_by_name || ""),
    },
    // {
    //   title: "Updated By",
    //   dataIndex: "updated_by_name",
    //   key: "updated_by_name",
    //   sorter: (a, b) => (a.updated_by_name || "").localeCompare(b.updated_by_name || ""),
    // },
    {
      title: "Approved By",
      dataIndex: "approved_by_name",
      key: "approved_by_name",
      sorter: (a, b) => (a.approved_by_name || "").localeCompare(b.approved_by_name || ""),
    },
    {
      title: "Download",
      key: "download",
      render: (_, record) => (
        <Button
          icon={<DownloadOutlined />}
          onClick={() => handleDownload(record.supplier_id, record.product_id)}
          size="small"
          className="text-blue-600"
        >
          Download
        </Button>
      ),
    },
    {
      title: "Actions",
      key: "actions",
      fixed: "right",
      render: (_, record) => (
        <Space size="small">
          <Tooltip title="View">
            <Button
              icon={<EyeOutlined />}
              onClick={() => setViewingSupplier(record)}
              size="small"
              className="text-blue-600"
            />
          </Tooltip>
          <Tooltip title="Edit">
            <Button
              icon={<EditOutlined />}
              onClick={() => handleEdit(record)}
              size="small"
              type="primary"
              className="bg-blue-600 hover:bg-blue-700"
            />
          </Tooltip>
          <Tooltip title="Delete">
            <Button
              icon={<DeleteOutlined />}
              onClick={() => handleDelete(record.supplier_id, record.product_id)}
              danger
              size="small"
              className="text-red-600"
            />
          </Tooltip>
        </Space>
      ),
    },
  ], [data, filteredData, selectedRows, renderTag, handleDownload, handleEdit, handleDelete]);

  const allColumns = useMemo(() =>
    columns.map((col) => ({
      key: col.key as string,
      label: typeof col.title === "string" ? col.title : col.key as string,
    })), [columns]);

  const columnKeys = useMemo(() => allColumns.map((c) => c.key), [allColumns]);
  const labelsMap = useMemo(() => Object.fromEntries(allColumns.map((c) => [c.key, c.label])), [allColumns]);
  const [visibleColumnKeys, setVisibleColumnKeys] = useState<string[]>(columnKeys);

  const handleVisibleColumnChange = useCallback((cols: string[]) => {
    setVisibleColumnKeys(cols);
  }, []);

  const filteredColumns = useMemo(() =>
    columns.filter((col) => typeof col.key === "string" && visibleColumnKeys.includes(col.key)),
    [columns, visibleColumnKeys]
  );

  // const SupplierCard = useCallback(({ supplier }: { supplier: SupplierApproval }) => (
  //   <div className="border rounded-xl p-4 shadow-sm hover:shadow-md bg-white dark:bg-gray-900 transition-all w-full">
  //     <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-2 gap-2">
  //       <h3 className="text-base sm:text-lg font-semibold text-gray-800 dark:text-white truncate">{supplier.supplier_name || "-"}</h3>
  //       <Tag color="blue">{supplier.status}</Tag>
  //     </div>
  //     <p className="text-sm text-gray-600 dark:text-gray-300 mb-1 truncate"><strong>Supplier ID:</strong> {supplier.supplier_id}</p>
  //     <p className="text-sm text-gray-600 dark:text-gray-300 mb-1 truncate"><strong>Supplier Type:</strong> {supplier.supplier_type || "-"}</p>
  //     <p className="text-sm text-gray-600 dark:text-gray-300 mb-1 truncate"><strong>Contact Person:</strong> {supplier.contactPerson || "-"}</p>
  //     <p className="text-sm text-gray-600 dark:text-gray-300 mb-1 truncate"><strong>Product:</strong> {supplier.product_name || "-"}</p>
  //     <p className="text-sm text-gray-600 dark:text-gray-300 mb-1 truncate"><strong>Product Code:</strong> {supplier.product_code || "-"}</p>
  //     <p className="text-sm text-gray-600 dark:text-gray-300 mb-1"><strong>Risk:</strong> {renderTag(supplier.risk_level || "")}</p>
  //     <p className="text-sm text-gray-600 dark:text-gray-300 mb-1"><strong>Status:</strong> {renderTag(supplier.status || "")}</p>
  //     <p className="text-sm text-gray-600 dark:text-gray-300 mb-1 truncate"><strong>Tested By:</strong> {supplier.testedBy || "-"}</p>
  //     <p className="text-sm text-gray-600 dark:text-gray-300 mb-1 truncate"><strong>Test Date:</strong> {supplier.testDate || "-"}</p>
  //     <div className="mt-2 flex gap-2">
  //       <Tooltip title="View">
  //         <Button icon={<EyeOutlined />} onClick={() => setViewingSupplier(supplier)} size="small" className="text-blue-600" />
  //       </Tooltip>
  //       <Tooltip title="Edit">
  //         <Button icon={<EditOutlined />} onClick={() => handleEdit(supplier)} size="small" className="text-blue-600" />
  //       </Tooltip>
  //       <Tooltip title="Delete">
  //         <Button icon={<DeleteOutlined />} onClick={() => handleDelete(supplier.supplier_id, supplier.product_id)} size="small" className="text-red-600" />
  //       </Tooltip>
  //       <Tooltip title="Download">
  //         <Button icon={<DownloadOutlined />} onClick={() => handleDownload(supplier.supplier_id, supplier.product_id)} size="small" className="text-blue-600">
  //           Download
  //         </Button>
  //       </Tooltip>
  //     </div>
  //   </div>
  // ), [renderTag, handleEdit, handleDelete, handleDownload]);
const SupplierCard = React.memo(({ supplier }: { supplier: SupplierApproval }) => (
  <GlassCard gradient blurStrength="lg" hoverLift color="success">
    <div className="border rounded-xl p-4 shadow-sm hover:shadow-md bg-white dark:bg-gray-900 
                    transition-all w-full flex flex-col">

      {/* Header */}
      <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-4">
        <h3 className="text-base sm:text-lg font-semibold text-gray-800 dark:text-white truncate">
          {supplier.supplier_name || "-"}
        </h3>
        <Tag color="blue">{supplier.status || "N/A"}</Tag>
      </div>

      {/* Line by Line Info */}
      <div className="text-sm space-y-2">
        <p className="text-gray-600 dark:text-gray-300"><strong>Supplier ID:</strong> {supplier.supplier_id}</p>
        <p className="text-gray-600 dark:text-gray-300"><strong>Supplier Type:</strong> {supplier.supplier_type || "-"}</p>
        <p className="text-gray-600 dark:text-gray-300"><strong>Contact Person:</strong> {supplier.contactPerson || "-"}</p>
        <p className="text-gray-600 dark:text-gray-300"><strong>Email:</strong> {supplier.email || "N/A"}</p>
        <p className="text-gray-600 dark:text-gray-300"><strong>Phone:</strong> {supplier.phone_number || "N/A"}</p>
        <p className="text-gray-600 dark:text-gray-300"><strong>Product:</strong> {supplier.product_name || "-"}</p>
        <p className="text-gray-600 dark:text-gray-300"><strong>Product Code:</strong> {supplier.product_code || "-"}</p>
        <p className="text-gray-600 dark:text-gray-300"><strong>Risk Level:</strong> {renderTag(supplier.risk_level || "")}</p>
        <p className="text-gray-600 dark:text-gray-300"><strong>Inspection Result:</strong> {renderTag(supplier.inspectionResult || "")}</p>
        <p className="text-gray-600 dark:text-gray-300"><strong>Rejection Reason:</strong> {supplier.rejectionReason || "N/A"}</p>
        <p className="text-gray-600 dark:text-gray-300"><strong>Test Result Details:</strong> {supplier.testResultDetails || "N/A"}</p>
        <p className="text-gray-600 dark:text-gray-300"><strong>Tested By:</strong> {supplier.testedBy || "-"}</p>
        <p className="text-gray-600 dark:text-gray-300"><strong>Test Date:</strong> {supplier.testDate ? dayjs(supplier.testDate).format("YYYY-MM-DD") : "-"}</p>
        <p className="text-gray-600 dark:text-gray-300"><strong>Approved At:</strong> {supplier.approved_at ? dayjs(supplier.approved_at).format("YYYY-MM-DD") : "N/A"}</p>
        <p className="text-gray-600 dark:text-gray-300"><strong>Approved By:</strong> {supplier.approved_by_name || "N/A"}</p>
        <p className="text-gray-500 dark:text-gray-400 text-xs"><strong>Created:</strong> {supplier.created_at ? dayjs(supplier.created_at).format("YYYY-MM-DD") : "N/A"}</p>
        <p className="text-gray-500 dark:text-gray-400 text-xs"><strong>Updated:</strong> {supplier.updated_at ? dayjs(supplier.updated_at).format("YYYY-MM-DD") : "N/A"}</p>
      </div>

      {/* Footer Actions */}
      <div className="mt-3 flex flex-wrap gap-2">
        <Tooltip title="View">
          <Button
            icon={<EyeOutlined />}
            onClick={() => setViewingSupplier(supplier)}
            size="small"
            className="text-blue-600"
          />
        </Tooltip>
        <Tooltip title="Edit">
          <Button
            icon={<EditOutlined />}
            onClick={() => handleEdit(supplier)}
            size="small"
            className="text-blue-600"
          />
        </Tooltip>
        <Tooltip title="Delete">
          <Button
            icon={<DeleteOutlined />}
            onClick={() => handleDelete(supplier.supplier_id, supplier.product_id)}
            size="small"
            className="text-red-600"
          />
        </Tooltip>
        <Tooltip title="Download">
          <Button
            icon={<DownloadOutlined />}
            onClick={() => handleDownload(supplier.supplier_id, supplier.product_id)}
            size="small"
            className="text-blue-600"
          >
            Download
          </Button>
        </Tooltip>
      </div>
    </div>
  </GlassCard>
));
SupplierCard.displayName = "SupplierCard";



  const total = data.length;
  const approved = data.filter((s) => s.status === "Approved").length;
  const rejected = data.filter((s) => s.status === "Rejected").length;
  const pending = data.filter((s) => s.status === "Submitted" || s.status === "Draft").length;
  const highRisk = data.filter((s) => s.risk_level === "High").length;

  const trendData = [
    { value: 10 },
    { value: 15 },
    { value: 12 },
    { value: 18 },
    { value: 20 },
  ];

  return (
    <>
      <div className="px-2 sm:px-2 md:px-2 lg:px-2 md:py-2 lg:py-2 py-2 sm:py-2">
        <h2 className="flex items-center gap-2 text-xl sm:text-2xl md:text-3xl font-bold text-gray-800 dark:text-white mb-2">
  <ShieldCheckIcon className="w-7 h-7 text-green-600 dark:text-green-400 flex-shrink-0" />
  <span>Supplier Approvals</span>
  <span className="text-gray-500 text-base sm:text-lg font-medium">
    ({data.length})
  </span>
</h2>
        <p className="text-gray-600 dark:text-gray-300 text-sm sm:text-base mb-4">
          View and monitor supplier approval statuses.
        </p>
        <Breadcrumb
          className="mb-4 sm:mb-6 mt-2 text-sm text-gray-500 dark:text-gray-400"
          separator=">"
          items={[
            {
              title: (
                <Tooltip title="Dashboard">
                  <HomeOutlined />
                </Tooltip>
              ),
              href: "/dashboard",
            },
            {
              title: "Master Data",
            },
            {
              title: "Supplier Approvals",
            },
          ]}
        />

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mt-3 mb-6">
          <WidgetCard
            title="Total Approvals"
            value={total}
            icon={<AppstoreOutlined />}
            color="green"
            percentage={12}
            description="All supplier approval records"
            trendData={trendData}
          />
          <WidgetCard
            title="Approved"
            value={approved}
            icon={<CheckCircleTwoTone twoToneColor="#52c41a" />}
            color="blue"
            percentage={8}
            description="Suppliers with approved status"
            trendData={trendData}
          />
          <WidgetCard
            title="Rejected"
            value={rejected}
            icon={<StopOutlined />}
            color="red"
            percentage={-3}
            description="Suppliers with rejected status"
            trendData={trendData}
          />
          <WidgetCard
            title="Pending Approvals"
            value={pending}
            icon={<WarningOutlined />}
            color="orange"
            percentage={5}
            description="Suppliers awaiting approval"
            trendData={trendData}
          />
        </div>

        <div className="mt-4">
          <Card title="Supplier Approval List" className="p-0 mb-6 shadow-sm">
            <div className="p-4 flex flex-col gap-4">
              <div className="flex flex-col sm:flex-row justify-between items-stretch sm:items-center gap-4">
                <Input
                  prefix={<SearchOutlined />}
                  allowClear
                  placeholder="Search Supplier, Product or ID"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="w-full sm:w-64 border-gray-300 dark:border-gray-600 rounded-md shadow-sm"
                />
                <Dropdown
                  className="w-full sm:w-auto border-gray-300 dark:border-gray-600"
                  menu={{
                    items: [
                      { label: "Export CSV", key: "csv" },
                      { label: "Export PDF", key: "pdf" },
                    ],
                  }}
                >
                  <Button icon={<DownloadOutlined />}>Export</Button>
                </Dropdown>
              </div>
              <div className="flex flex-col gap-4 lg:flex-row lg:justify-between lg:items-center">
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 w-full lg:w-auto">
                  <Select
                    placeholder="Risk Level"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({ ...prev, riskClassification: value }))
                    }
                    allowClear
                    options={riskOptions.map((v) => ({ label: v, value: v }))}
                  />
                  <Select
                    placeholder="Validation Status"
                    className="w-full max-w-[160px]"
                    onChange={(value) =>
                      setAdvancedFilters((prev) => ({ ...prev, validation: value }))
                    }
                    allowClear
                    options={validationOptions.map((v) => ({ label: v, value: v }))}
                  />
                </div>
                <div className="flex flex-col sm:flex-row flex-wrap gap-4 justify-start lg:justify-end w-full lg:w-auto">
                  <ColumnVisibilityToggle
                    columns={columnKeys}
                    labels={labelsMap}
                    storageKey="supplier_approval_table_visibility"
                    onChange={handleVisibleColumnChange}
                  />
                  <Button
                    onClick={handleBulkDelete}
                    disabled={selectedRows.length === 0}
                    danger
                    className="w-full sm:w-auto"
                  >
                    Bulk Delete
                  </Button>
                  <div className="hidden sm:block">
                    <Switch
                      checkedChildren={<AppstoreOutlined />}
                      unCheckedChildren={<TableOutlined />}
                      checked={isGridView}
                      onChange={setIsGridView}
                      className="w-full sm:w-auto"
                    />
                  </div>
                </div>
              </div>

              {(advancedFilters.riskClassification || advancedFilters.validation) && (
                <div className="flex flex-wrap gap-2 mt-2">
                  {advancedFilters.riskClassification && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          riskClassification: undefined,
                        }))
                      }
                    >
                      Risk Level: {advancedFilters.riskClassification}
                    </Tag>
                  )}
                  {advancedFilters.validation && (
                    <Tag
                      closable
                      onClose={() =>
                        setAdvancedFilters((prev) => ({
                          ...prev,
                          validation: undefined,
                        }))
                      }
                    >
                      Validation: {advancedFilters.validation}
                    </Tag>
                  )}
                </div>
              )}
            </div>

            {isGridView || isMobile ? (
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-3 gap-4 p-4">
                {paginatedData.map((supplier) => (
                  <SupplierCard key={`${supplier.supplier_id}-${supplier.product_id}`} supplier={supplier} />
                ))}
              </div>
            ) : (
              <div className="overflow-auto">
                <DataTable<SupplierApproval>
                  columns={filteredColumns}
                  dataSource={paginatedData}
                  rowKey={(record) => `${record.supplier_id}-${record.product_id}`}
                  scrollX="max-content"
                  className="mt-4"
                  pagination={false}
                  loading={loading}
                />
              </div>
            )}
            <div className="flex flex-col sm:flex-row justify-between items-center px-4 py-2 text-sm text-gray-600 dark:text-gray-300">
              <div className="mb-2 sm:mb-0">
                Showing <strong>{(currentPage - 1) * pageSize + 1}</strong> to{" "}
                <strong>{Math.min(currentPage * pageSize, filteredData.length)}</strong> of{" "}
                <strong>{filteredData.length}</strong> approvals
              </div>
              <Pagination
                current={currentPage}
                pageSize={pageSize}
                total={filteredData.length}
                onChange={(page, size) => {
                  setCurrentPage(page);
                  setPageSize(size);
                }}
                showSizeChanger
                pageSizeOptions={["10", "30", "50", "100"]}
                size="small"
              />
            </div>
          </Card>
        </div>

        <motion.div
          initial={{ opacity: 0, scale: 0.95 }}
          animate={{ opacity: 1, scale: 1 }}
          exit={{ opacity: 0, scale: 0.95 }}
        >
          <ConfirmDialog
            open={confirmDialog.open}
            title={confirmDialog.title}
            message={confirmDialog.message}
            onClose={() => setConfirmDialog({ ...confirmDialog, open: false })}
            onConfirm={confirmDialog.onConfirm}
          />
          <Modal
            title="Supplier Details"
            open={!!viewingSupplier}
            onCancel={() => setViewingSupplier(null)}
            footer={null}
            className="w-full max-w-[90vw] sm:max-w-[500px] top-4"
          >
            {viewingSupplier && (
              <div className="space-y-2 text-sm sm:text-base">
                <p><strong>Supplier ID:</strong> {viewingSupplier.supplier_id}</p>
                <p><strong>Supplier Name:</strong> <span className="text-green-600">{viewingSupplier.supplier_name || "-"}</span></p>
                <p><strong>Supplier Type:</strong> {viewingSupplier.supplier_type || "-"}</p>
                <p><strong>Contact Person:</strong> {viewingSupplier.contactPerson || "-"}</p>
                <p><strong>Email:</strong> {viewingSupplier.email || "-"}</p>
                <p><strong>Phone:</strong> {viewingSupplier.phone_number || "-"}</p>
                <p><strong>Product ID:</strong> {viewingSupplier.product_id}</p>
                <p><strong>Product Name:</strong> {viewingSupplier.product_name || "-"}</p>
                <p><strong>Product Code:</strong> {viewingSupplier.product_code || "-"}</p>
                <p><strong>Risk Level:</strong> {viewingSupplier.risk_level || "-"}</p>
                <p><strong>Inspection Result:</strong> {viewingSupplier.inspectionResult || "-"}</p>
                <p><strong>Tested By:</strong> {viewingSupplier.testedBy || "-"}</p>
                <p><strong>Test Date:</strong> {viewingSupplier.testDate || "-"}</p>
                <p><strong>Rejection Reason:</strong> {viewingSupplier.rejectionReason || "-"}</p>
                <p><strong>Test Result Details:</strong> {viewingSupplier.testResultDetails || "-"}</p>
                <p><strong>Created By:</strong> {viewingSupplier.created_by_name || "-"}</p>                
                <p><strong>Approved At:</strong> {viewingSupplier.approved_at || "-"}</p>
                <p><strong>Status:</strong> {viewingSupplier.status || "-"}</p>
                <p><strong>Created By:</strong> {viewingSupplier.created_by_name || "-"}</p>
                <p><strong>Updated By:</strong> {viewingSupplier.updated_by_name || "-"}</p>
                <p><strong>Approved By:</strong> {viewingSupplier.approved_by_name || "-"}</p>
                <p><strong>Created At:</strong> {viewingSupplier.created_at || "-"}</p>
              </div>
            )}
          </Modal>

          <Modal
            title="Edit Supplier Approval"
            open={!!editingSupplier}
            onOk={handleUpdateStatus}
            onCancel={() => setEditingSupplier(null)}
            okText="Update"
            okButtonProps={{ loading: isSubmitting }}
            className="w-full max-w-[90vw] sm:max-w-[500px] top-4"
          >
            {editingSupplier && (
              <div className="space-y-4">
                <p><strong>Supplier ID:</strong> {editingSupplier.supplier_id}</p>
                <p><strong>Supplier Name:</strong> {editingSupplier.supplier_name || "-"}</p>
                <p><strong>Supplier Type:</strong> {editingSupplier.supplier_type || "-"}</p>
                <p><strong>Contact Person:</strong> {editingSupplier.contactPerson || "-"}</p>
                <p><strong>Email:</strong> {editingSupplier.email || "-"}</p>
                <p><strong>Phone:</strong> {editingSupplier.phone_number || "-"}</p>
                <p><strong>Product ID:</strong> {editingSupplier.product_id}</p>
                <p><strong>Product Name:</strong> {editingSupplier.product_name || "-"}</p>
                <p><strong>Product Code:</strong> {editingSupplier.product_code || "-"}</p>
                <p><strong>Risk Level:</strong> {editingSupplier.risk_level || "-"}</p>
                <p><strong>Inspection Result:</strong> {editingSupplier.inspectionResult || "-"}</p>
                <p><strong>Tested By:</strong> {editingSupplier.testedBy || "-"}</p>
                <p><strong>Test Date:</strong> {editingSupplier.testDate || "-"}</p>
                <p><strong>Rejection Reason:</strong> {editingSupplier.rejectionReason || "-"}</p>
                <p><strong>Test Result Details:</strong> {editingSupplier.testResultDetails || "-"}</p>
                <p><strong>Approved At:</strong> {editingSupplier.approved_at || "-"}</p>

                <p><strong>Created By:</strong> {editingSupplier.created_by_name || "-"}</p>
                {/* <p><strong>Updated By:</strong> {editingSupplier.updated_by_name || "-"}</p> */}
                <p><strong>Approved By:</strong> {editingSupplier.approved_by_name || "-"}</p>
                <p><strong>Created At:</strong> {editingSupplier.created_at || "-"}</p>
                <p><strong>Status:</strong></p>
                <Select
                  placeholder="Status"
                  value={editingSupplier.status}
                  onChange={(value) =>
                    setEditingSupplier((prev) => (prev ? { ...prev, status: value } : null))
                  }
                  className="w-full"
                  options={validationOptions.map((v) => ({ label: v, value: v }))}
                />
              </div>
            )}
          </Modal>
        </motion.div>
      </div>
    </>
  );
}