// 'use client';

// import { LineChart, Line, ResponsiveContainer, AreaChart, Area } from 'recharts';
// import { ReactNode } from 'react';
// import clsx from 'clsx';
// import CountUp from 'react-countup';


// interface WidgetCardProps {
//   title: string;
//   value: number;
//   icon?: ReactNode;
//   trendData?: { value: number }[];
//   color?: 'green' | 'blue' | 'red' | 'orange' | 'gray' | 'gold';
//   description?: string;
//   className?: string;
//   percentage?: number; // e.g., +12% growth
// }

// const getColor = (color: string) => {
//   switch (color) {
//     case 'green':
//       return {
//         text: 'text-green-600',
//         bg: 'bg-green-100 dark:bg-green-900/40',
//         stroke: '#22c55e',
//         badge: 'bg-green-200 text-green-800',
//       };
//     case 'blue':
//       return {
//         text: 'text-blue-600',
//         bg: 'bg-blue-100 dark:bg-blue-900/40',
//         stroke: '#3b82f6',
//         badge: 'bg-blue-200 text-blue-800',
//       };
//     case 'red':
//       return {
//         text: 'text-red-600',
//         bg: 'bg-red-100 dark:bg-red-900/40',
//         stroke: '#ef4444',
//         badge: 'bg-red-200 text-red-800',
//       };
//     case 'orange':
//       return {
//         text: 'text-orange-600',
//         bg: 'bg-orange-100 dark:bg-orange-900/40',
//         stroke: '#f97316',
//         badge: 'bg-orange-200 text-orange-800',
//       };
//     default:
//       return {
//         text: 'text-gray-700 dark:text-gray-200',
//         bg: 'bg-gray-100 dark:bg-gray-800/60',
//         stroke: '#6b7280',
//         badge: 'bg-gray-200 text-gray-800',
//       };
//   }
// };

// export default function WidgetCard({
//   title,
//   value,
//   icon,
//   trendData,
//   color = 'green',
//   description,
//   className,
//   percentage,
// }: WidgetCardProps) {
//   const colors = getColor(color);

//   return (
//     <div
//   className={clsx(
//     'rounded-xl p-4 bg-white dark:bg-gray-900 border border-gray-200 dark:border-gray-700 flex flex-col justify-between',
//     'transition-all duration-300 ease-in-out transform hover:scale-[1.02] hover:shadow-lg',
//     className
//   )}
// >

//       <div className="flex items-start justify-between mb-2">
//         <div>
//           <h4 className="text-sm font-medium text-gray-500 dark:text-gray-400">{title}</h4>
//           <p className={`text-3xl font-bold ${colors.text}`}>
//   <CountUp end={value} duration={1.5} separator="," />
// </p>
//         </div>
//         {icon && (
//           <div className={`p-2 rounded-full ${colors.bg} ${colors.text}`}>
//             {icon}
//           </div>
//         )}
//       </div>

//       {description && (
//         <p className="text-xs text-gray-500 dark:text-gray-400 mb-2">{description}</p>
//       )}

//       {percentage !== undefined && (
//         <span
//           className={`mt-1 inline-block text-xs font-medium px-2 py-1 rounded ${colors.badge}`}
//         >
//           {percentage > 0 ? '+' : ''}
//           {percentage}% from last month
//         </span>
//       )}
//       {trendData && (
//         <div className="h-12 mt-2">
//           <ResponsiveContainer width="100%" height="100%">
//   <AreaChart data={trendData}>
//     <defs>
//       <linearGradient id="colorTrend" x1="0" y1="0" x2="0" y2="1">
//         <stop offset="0%" stopColor={colors.stroke} stopOpacity={0.4} />
//         <stop offset="100%" stopColor={colors.stroke} stopOpacity={0.05} />
//       </linearGradient>
//     </defs>
//     <Area
//       type="monotone"
//       dataKey="value"
//       stroke={colors.stroke}
//       fill="url(#colorTrend)"
//       strokeWidth={2}
//       dot={false}
//     />
//   </AreaChart>
// </ResponsiveContainer>

//         </div>
//       )}
//     </div>
//   );
// }
'use client';

import React, { useMemo } from 'react';
import dynamic from 'next/dynamic';
import { ReactNode } from 'react';
import clsx from 'clsx';
import CountUp from 'react-countup';
import { FaArrowUp, FaArrowDown } from 'react-icons/fa';
import { Tooltip } from 'recharts';

// Dynamically import Recharts components to reduce initial bundle size
const ResponsiveContainer = dynamic(
  () => import('recharts').then(mod => mod.ResponsiveContainer),
  { ssr: false }
);
const AreaChart = dynamic(() => import('recharts').then(mod => mod.AreaChart), {
  ssr: false,
});
const Area = dynamic(() => import('recharts').then(mod => mod.Area), {
  ssr: false,
});


interface WidgetCardProps {
  title: string;
  value: number;
  icon?: ReactNode;
  trendData?: { value: number }[];
  color?: 'green' | 'blue' | 'red' | 'orange' | 'gray' | 'gold' | 'purple';
  description?: string;
  className?: string;
  percentage?: number; // e.g., +12% growth
  onClick?: () => void;
  onTrendClick?: () => void;
  loading?: boolean; // for skeleton loader
}

const getColor = (color: string) => {
  switch (color) {
    case 'green':
      return {
        text: 'text-green-600',
        bg: 'bg-green-100 dark:bg-green-900/40',
        stroke: '#22c55e',
        badgePositive: 'bg-green-200 text-green-800',
        badgeNegative: 'bg-red-200 text-red-800',
      };
    case 'blue':
      return {
        text: 'text-blue-600',
        bg: 'bg-blue-100 dark:bg-blue-900/40',
        stroke: '#3b82f6',
        badgePositive: 'bg-blue-200 text-blue-800',
        badgeNegative: 'bg-red-200 text-red-800',
      };
    case 'red':
      return {
        text: 'text-red-600',
        bg: 'bg-red-100 dark:bg-red-900/40',
        stroke: '#ef4444',
        badgePositive: 'bg-red-200 text-red-800',
        badgeNegative: 'bg-green-200 text-green-800',
      };
    case 'orange':
      return {
        text: 'text-orange-600',
        bg: 'bg-orange-100 dark:bg-orange-900/40',
        stroke: '#f97316',
        badgePositive: 'bg-orange-200 text-orange-800',
        badgeNegative: 'bg-red-200 text-red-800',
      };
    default:
      return {
        text: 'text-gray-700 dark:text-gray-200',
        bg: 'bg-gray-100 dark:bg-gray-800/60',
        stroke: '#6b7280',
        badgePositive: 'bg-gray-200 text-gray-800',
        badgeNegative: 'bg-red-200 text-red-800',
      };
  }
};

const CountUpMemo = React.memo(CountUp);

export default function WidgetCard({
  title,
  value,
  icon,
  trendData,
  color = 'green',
  description,
  className,
  percentage,
  onClick,
  onTrendClick,
  loading = false,
}: WidgetCardProps) {
  // Memoize colors for performance
  const colors = useMemo(() => getColor(color), [color]);
  const { text, bg, stroke, badgePositive, badgeNegative } = colors;

  // Determine badge color based on percentage sign
  const badgeColor =
    percentage !== undefined && percentage > 0 ? badgePositive : badgeNegative;

  // Limit trendData to last 20 points if large (preprocessing suggestion)
  const limitedTrendData = useMemo(() => {
    if (!trendData) return [];
    return trendData.length > 20
      ? trendData.slice(trendData.length - 20)
      : trendData;
  }, [trendData]);

  // Simple skeleton loader for loading state
  if (loading) {
    return (
      <div
        className={clsx(
          'rounded-xl p-4 bg-white dark:bg-gray-900 border border-gray-200 dark:border-gray-700',
          className
        )}
      >
        <div className="h-6 bg-gray-300 dark:bg-gray-700 rounded w-1/3 mb-2 animate-pulse" />
        <div className="h-12 bg-gray-300 dark:bg-gray-700 rounded w-full animate-pulse" />
      </div>
    );
  }

  return (
    <div
      onClick={onClick}
      role={onClick ? 'button' : undefined}
      tabIndex={onClick ? 0 : undefined}
      aria-label={title}
      className={clsx(
        'rounded-xl p-4 bg-white dark:bg-gray-900 border border-gray-200 dark:border-gray-700 flex flex-col justify-between',
        'transition-all duration-300 ease-in-out transform hover:scale-[1.02] hover:shadow-lg',
        className,
        onClick && 'cursor-pointer focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500'
      )}
    >
      <div className="flex items-start justify-between mb-2">
        <div>
          <h4 className="text-sm font-semibold text-gray-700 dark:text-gray-300">
            {title}
          </h4>
          <p className={`text-3xl font-extrabold ${text}`}>
            <CountUpMemo end={value} duration={1.5} separator="," />
          </p>
        </div>
        {icon && <div className={`p-2 rounded-full ${bg} ${text}`}>{icon}</div>}
      </div>

      {description && (
        <p className="text-xs text-gray-500 dark:text-gray-400 mb-2">{description}</p>
      )}

      {percentage !== undefined && (
        <span
          className={`mt-1 inline-flex items-center text-xs font-medium px-2 py-1 rounded ${badgeColor}`}
        >
          {percentage > 0 ? (
            <FaArrowUp className="mr-1" aria-hidden="true" />
          ) : (
            <FaArrowDown className="mr-1" aria-hidden="true" />
          )}
          {percentage > 0 ? '+' : ''}
          {percentage}% from last month
        </span>
      )}

      {limitedTrendData.length > 0 && (
        <div
          className="h-12 mt-2"
          onClick={onTrendClick}
          role={onTrendClick ? 'button' : undefined}
          tabIndex={onTrendClick ? 0 : undefined}
          aria-label={`${title} trend chart`}
        >
          <ResponsiveContainer width="100%" height="100%">
            <AreaChart data={limitedTrendData}>
              <defs>
                <linearGradient id="colorTrend" x1="0" y1="0" x2="0" y2="1">
                  <stop offset="0%" stopColor={stroke} stopOpacity={0.4} />
                  <stop offset="100%" stopColor={stroke} stopOpacity={0.05} />
                </linearGradient>
              </defs>
              <Area
                type="monotone"
                dataKey="value"
                stroke={stroke}
                fill="url(#colorTrend)"
                strokeWidth={2}
                dot={false}
                isAnimationActive={true}
                animationDuration={1000}
              />
              <Tooltip
                contentStyle={{ backgroundColor: 'transparent', border: 'none', color: text }}
                   formatter={(value: number | string, _name: string) => [value, '']}

                cursor={false}
              />
            </AreaChart>
          </ResponsiveContainer>
        </div>
      )}
    </div>
  );
}
