"use client";

import { useState, useEffect } from "react";
import { Button, Dropdown, Checkbox, Divider } from "antd";
import { EyeOutlined } from "@ant-design/icons";

interface ColumnVisibilityToggleProps<T extends string> {
  columns: T[];
  storageKey: string;
  labels?: Record<T, string>;
  onChange: (cols: T[]) => void;
  className?: string; // Added className prop
}

export default function ColumnVisibilityToggle<T extends string>({
  columns,
  storageKey,
  labels,
  onChange,
  className,
}: ColumnVisibilityToggleProps<T>) {
  const [visibleColumns, setVisibleColumns] = useState<T[]>([]);
  const [open, setOpen] = useState(false);

  // Initialize once
  useEffect(() => {
    setVisibleColumns(columns);
    onChange(columns);
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []); // only run once on mount

  const toggleColumn = (col: T) => {
    const updated = visibleColumns.includes(col)
      ? visibleColumns.filter((c) => c !== col)
      : [...visibleColumns, col];

    setVisibleColumns(updated);
    onChange(updated);
  };

  const handleSelectAll = (checked: boolean) => {
    const updated = checked ? [...columns] : [];
    setVisibleColumns(updated);
    onChange(updated);
  };

  return (
    <Dropdown
      open={open}
      onOpenChange={setOpen}
      trigger={["click"]}
      placement="bottomRight"
    popupRender={() => (
  <div className="bg-white dark:bg-gray-800 rounded-md shadow-md border border-gray-200 dark:border-gray-600 w-150 p-4 text-sm">
    {/* Column checkboxes in a 2-column grid */}
    <div className="grid grid-cols-3 gap-x-6 gap-y-2">
      {columns.map((col) => (
        <Checkbox
          key={col}
          checked={visibleColumns.includes(col)}
          onChange={() => toggleColumn(col)}
          className="dark:text-gray-100"
        >
          {labels?.[col] ?? col}
        </Checkbox>
      ))}
    </div>

    <Divider className="my-3 dark:border-gray-600" />

    {/* Select All */}
    <div className="mb-3">
      <Checkbox
        indeterminate={
          visibleColumns.length > 0 &&
          visibleColumns.length < columns.length
        }
        checked={visibleColumns.length === columns.length}
        onChange={(e) => handleSelectAll(e.target.checked)}
        className="dark:text-gray-100"
      >
        Select All
      </Checkbox>
    </div>

    <div className="text-right">
      <Button
        size="small"
        onClick={() => setOpen(false)}
        className="dark:bg-gray-700 dark:text-gray-100 dark:hover:bg-gray-600 dark:border-gray-600"
      >
        Done
      </Button>
    </div>
  </div>
)}

    >
      <Button
        icon={<EyeOutlined />}
        className={`shadow-sm dark:bg-gray-700 dark:text-gray-100 dark:hover:bg-gray-600 dark:border-gray-600 ${className}`}
      >
        Manage Columns
         <span className="text-xs text-gray-500 ml-2">
  ({visibleColumns.length}/{columns.length})
</span>
      </Button>
    </Dropdown>
  );
}