// lib/api/training-feedback.ts
import axios from "axios";
import { API_BASE_PATH } from "@/config/apiConfig";
import { toast } from "sonner";

const API_BASE_URL = API_BASE_PATH;

/* ================================
   🟦 INTERFACES
================================ */

// CREATE payload
export interface TrainingFeedback {
  session_code: string;
  conducted_by: string;
  reviewed_by?: string;
  program_attended: string;
  venue?: string | null;
  date: string;

  // Ratings (included in create)
  training_content_rating: string;
  training_method_rating: string;
  trainer_effectiveness: string;
  duration_adequacy: string;

  // Booleans
  expectations_satisfied: boolean;
  new_skills_gained: boolean;
  applicability_to_job: boolean;
  further_training_needed?: boolean;

  comments?: string | null;
}

// CREATE response
export interface CreateTrainingFeedbackResponse {
  message: string;
  feedback_code: string;
}

// FULL DATA returned by GET endpoints
export interface TrainingFeedbackData {
  feedback_code: string;
  session_code: string;

  employee_code?: string;
  employee_name?: string;

  conducted_by?: string;
  reviewed_by?: string;

  conducted_by_name?: string;
  conducted_by_id?: string;
  reviewed_by_name?: string;
  reviewed_by_id?: string;

  program_attended: string;
  venue?: string | null;
  date: string;

  training_content_rating: string;
  training_method_rating: string;
  trainer_effectiveness: string;
  duration_adequacy: string;

  expectations_satisfied: boolean;
  new_skills_gained: boolean;
  applicability_to_job: boolean;
  further_training_needed: boolean;

  comments?: string | null;

  created_at?: string;
  updated_at?: string;
}

// GET ALL
export type GetAllTrainingFeedbackResponse = TrainingFeedbackData[];

// GET BY CODE
export type GetTrainingFeedbackByCodeResponse = TrainingFeedbackData;

// UPDATE FEEDBACK info (PUT /update_training_feedback)
export interface UpdateTrainingFeedbackResponse {
  message: string;
  feedback_code: string;
}

// UPDATE RATINGS (PUT /update_feedback_ratings)
export interface UpdateFeedbackRatingsResponse {
  message: string;
}

// DELETE RESPONSE
export interface DeleteTrainingFeedbackResponse {
  message: string;
}

/* ================================
   🟦 AUTH HEADERS
================================ */

const getAuthHeaders = () => {
  let token = localStorage.getItem("authToken") || "";

  if (!token.startsWith("Bearer ")) {
    token = `Bearer ${token}`;
  }

  return {
    headers: {
      Authorization: token,
      "Content-Type": "application/x-www-form-urlencoded",
      Accept: "application/json",
    },
  };
};

/* ================================
   🟦 FORM BUILDERS
================================ */

// Used for CREATE + UPDATE FEEDBACK INFO
const buildTrainingFeedbackForm = (
  data: Partial<TrainingFeedback>
): URLSearchParams => {
  return new URLSearchParams({
    ...(data.session_code ? { session_code: data.session_code } : {}),
    ...(data.conducted_by ? { conducted_by: data.conducted_by } : {}),
    reviewed_by: data.reviewed_by || "",
    program_attended: data.program_attended || "",
    venue: data.venue || "",
    date: data.date || "",
    comments: data.comments || "",
  });
};

// Used for UPDATE RATINGS only
const buildRatingsForm = (data: Partial<TrainingFeedback>): URLSearchParams => {
  return new URLSearchParams({
    training_content_rating: data.training_content_rating || "",
    training_method_rating: data.training_method_rating || "",
    trainer_effectiveness: data.trainer_effectiveness || "",
    duration_adequacy: data.duration_adequacy || "",
    expectations_satisfied: String(data.expectations_satisfied),
    new_skills_gained: String(data.new_skills_gained),
    applicability_to_job: String(data.applicability_to_job),
    further_training_needed: String(data.further_training_needed),
  });
};

/* ================================
   🟦 ERROR HANDLER
================================ */

const showBackendError = (error: any) => {
  const msg =
    error?.response?.data?.detail ||
    error?.response?.data?.message ||
    error?.response?.data?.error ||
    error?.message ||
    "Something went wrong";

  toast.error(msg);
};

/* ================================
   🟦 API FUNCTIONS
================================ */

// CREATE
export const createTrainingFeedback = async (
  data: TrainingFeedback
): Promise<CreateTrainingFeedbackResponse> => {
  try {
    const response = await axios.post(
      `${API_BASE_URL}/create_training_feedback`,
      buildTrainingFeedbackForm(data),
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET ALL
export const getAllTrainingFeedbacks = async (): Promise<
  TrainingFeedbackData[]
> => {
  try {
    const response = await axios.get<GetAllTrainingFeedbackResponse>(
      `${API_BASE_URL}/get_all`,
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET BY CODE
export const getTrainingFeedbackByCode = async (
  feedbackCode: string
): Promise<TrainingFeedbackData> => {
  try {
    const response = await axios.get<GetTrainingFeedbackByCodeResponse>(
      `${API_BASE_URL}/training_feedback/${feedbackCode}`,
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// UPDATE FEEDBACK INFO (conducted_by, reviewed_by, program_attended, etc.)
export const updateTrainingFeedback = async (
  feedbackCode: string,
  data: Partial<TrainingFeedback>
): Promise<UpdateTrainingFeedbackResponse> => {
  try {
    const response = await axios.put(
      `${API_BASE_URL}/update_training_feedback/${feedbackCode}`,
      buildTrainingFeedbackForm(data),
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// UPDATE ONLY RATINGS + BOOLEAN FIELDS
export const updateFeedbackRatings = async (
  feedbackCode: string,
  data: Partial<TrainingFeedback>
): Promise<UpdateFeedbackRatingsResponse> => {
  try {
    const response = await axios.put(
      `${API_BASE_URL}/update_feedback_ratings/${feedbackCode}`,
      buildRatingsForm(data),
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// DELETE
export const deleteTrainingFeedback = async (
  feedbackCode: string
): Promise<DeleteTrainingFeedbackResponse> => {
  try {
    const response = await axios.delete(
      `${API_BASE_URL}/delete_feedback/${feedbackCode}`,
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};
