//app/api/supplier-rating.ts
import axios from "axios";
import { API_BASE_PATH } from "@/config/apiConfig";
import { toast } from "sonner";

const API_BASE_URL = API_BASE_PATH;

// Supplier Rating Interface (exactly as you wrote – NOT touched)
export interface SupplierRating {
  supplier_id: string;
  materialDescription?: string | null;
  lotReceived: number;
  lotAccepted: number;
  qualityRating: number;
  deliveryRating: number;
  serviceRating: number;
  priceRating: number;
  ratingDate: string;
  onTimeDelivery: number;
  shortDelivery: number;
  acceptedPercent: number;
  overallScore: number;
  ratingScale?: string;
  risklevel?: string;
  remark?: string;
  category?: string;
  rating_id?: string;
  created_at?: string;
  updated_at?: string;
  rating_By?: string;
  approved_id?: string | null;
  approved_By?: string | null;
}

// Auth headers (your original)
const getAuthHeaders = () => {
  const token = localStorage.getItem("authToken") || "";
  return {
    headers: {
      Authorization: token,
      "Content-Type": "application/x-www-form-urlencoded",
      Accept: "application/json",
    },
  };
};

// Show exact backend error only
const showBackendError = (error: any) => {
  const msg =
    error?.response?.data?.detail ||
    error?.response?.data?.message ||
    error?.message ||
    "Something went wrong";
  toast.error(msg);
};

// CREATE SUPPLIER RATING
export const createSupplierRating = async (data: SupplierRating) => {
  try {
    const form = new URLSearchParams({
      supplier_id: data.supplier_id,
      materialDescription: data.materialDescription ?? "",
      lotReceived: String(data.lotReceived),
      lotAccepted: String(data.lotAccepted),
      qualityRating: String(data.qualityRating),
      deliveryRating: String(data.deliveryRating),
      serviceRating: String(data.serviceRating),
      priceRating: String(data.priceRating),
      ratingDate: data.ratingDate,
      onTimeDelivery: String(data.onTimeDelivery),
      shortDelivery: String(data.shortDelivery),
      acceptedPercent: String(data.acceptedPercent),
      overallScore: String(data.overallScore),
      ratingScale: data.ratingScale ?? "",
      remark: data.remark ?? "",
    });

    const response = await axios.post(
      `${API_BASE_URL}/create-supplier-rating`,
      form,
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET ALL SUPPLIER RATINGS
export const getAllSupplierRatings = async (): Promise<SupplierRating[]> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/get-supplier-ratings`,
      getAuthHeaders()
    );
    return response.data.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// GET SUPPLIER RATING BY ID
export const getSupplierRatingById = async (
  supplierId: string
): Promise<SupplierRating> => {
  try {
    const response = await axios.get(
      `${API_BASE_URL}/get-supplier-ratings/${supplierId}`,
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// UPDATE SUPPLIER RATING
export const updateSupplierRating = async (
  supplierId: string,
  data: SupplierRating
) => {
  try {
    const form = new URLSearchParams({
      supplier_id: data.supplier_id,
      materialDescription: data.materialDescription ?? "",
      lotReceived: String(data.lotReceived),
      lotAccepted: String(data.lotAccepted),
      qualityRating: String(data.qualityRating),
      deliveryRating: String(data.deliveryRating),
      serviceRating: String(data.serviceRating),
      priceRating: String(data.priceRating),
      ratingDate: data.ratingDate,
      onTimeDelivery: String(data.onTimeDelivery),
      shortDelivery: String(data.shortDelivery),
      acceptedPercent: String(data.acceptedPercent),
      overallScore: String(data.overallScore),
      ratingScale: data.ratingScale ?? "",
      remark: data.remark ?? "",
    });

    const response = await axios.put(
      `${API_BASE_URL}/update-supplier-rating/${supplierId}`,
      form,
      getAuthHeaders()
    );
    return {
      ...response.data.detail,
      overallScore: response.data.detail.overallscore,
    };
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};

// DELETE SUPPLIER RATING
export const deleteSupplierRating = async (supplierId: string) => {
  try {
    const response = await axios.delete(
      `${API_BASE_URL}/delete-supplier-rating/${supplierId}`,
      getAuthHeaders()
    );
    return response.data;
  } catch (error: any) {
    showBackendError(error);
    throw error;
  }
};
